import { get } from 'lodash';
import { lastValueFrom } from 'rxjs';
import { usePluginContext } from '@grafana/data';
import { config } from '../config.mjs';
import '../services/index.mjs';
import { getBackendSrv } from '../services/backendSrv.mjs';

"use strict";
const baseURL = `/apis/userstorage.grafana.app/v0alpha1/namespaces/${config.namespace}/user-storage`;
async function apiRequest(requestOptions) {
  try {
    const { data: responseData, ...meta } = await lastValueFrom(
      getBackendSrv().fetch({
        ...requestOptions,
        url: baseURL + requestOptions.url,
        data: requestOptions.body,
        showErrorAlert: false
      })
    );
    return { data: responseData, meta };
  } catch (error) {
    return requestOptions.manageError ? requestOptions.manageError(error) : { error };
  }
}
class UserStorage {
  constructor(service) {
    this.service = service;
    this.userUID = config.bootData.user.uid === "" ? config.bootData.user.id.toString() : config.bootData.user.uid;
    this.resourceName = `${service}:${this.userUID}`;
    this.canUseUserStorage = config.bootData.user.isSignedIn;
  }
  async init() {
    if (this.storageSpec !== void 0) {
      return;
    }
    const userStorage = await apiRequest({
      url: `/${this.resourceName}`,
      method: "GET",
      manageError: (error) => {
        if (get(error, "status") === 404) {
          this.storageSpec = null;
          return { error: null };
        }
        return { error };
      }
    });
    if ("error" in userStorage) {
      return userStorage.error;
    }
    this.storageSpec = userStorage.data.spec;
    return;
  }
  async getItem(key) {
    if (!this.canUseUserStorage) {
      return localStorage.getItem(`${this.resourceName}:${key}`);
    }
    await this.init();
    if (!this.storageSpec) {
      return localStorage.getItem(`${this.resourceName}:${key}`);
    }
    return this.storageSpec.data[key];
  }
  async setItem(key, value) {
    if (!this.canUseUserStorage) {
      localStorage.setItem(`${this.resourceName}:${key}`, value);
      return;
    }
    const newData = { data: { [key]: value } };
    const error = await this.init();
    if (error) {
      localStorage.setItem(`${this.resourceName}:${key}`, value);
      return;
    }
    if (!this.storageSpec) {
      await apiRequest({
        url: `/`,
        method: "POST",
        body: {
          metadata: { name: this.resourceName, labels: { user: this.userUID, service: this.service } },
          spec: newData
        },
        manageError: (error2) => {
          localStorage.setItem(`${this.resourceName}:${key}`, value);
          return { error: error2 };
        }
      });
      this.storageSpec = newData;
      return;
    }
    this.storageSpec.data[key] = value;
    await apiRequest({
      headers: { "Content-Type": "application/merge-patch+json" },
      url: `/${this.resourceName}`,
      method: "PATCH",
      body: { spec: newData },
      manageError: (error2) => {
        localStorage.setItem(`${this.resourceName}:${key}`, value);
        return { error: error2 };
      }
    });
  }
}
function usePluginUserStorage() {
  const context = usePluginContext();
  if (!context) {
    throw new Error(`No PluginContext found. The usePluginUserStorage() hook can only be used from a plugin.`);
  }
  return new UserStorage(context == null ? void 0 : context.meta.id);
}

export { UserStorage, usePluginUserStorage };
//# sourceMappingURL=userStorage.mjs.map
