import { jsx, jsxs } from 'react/jsx-runtime';
import { cx } from '@emotion/css';
import debounce from 'debounce-promise';
import { useReducer, useCallback, useEffect, useMemo } from 'react';
import { selectors } from '@grafana/e2e-selectors';
import { t, Trans } from '@grafana/i18n';
import { useTheme2, Modal, Input, MultiSelect, Spinner, Toggletip, ButtonGroup, Button, Icon, Pagination } from '@grafana/ui';
import { getDebounceTimeInMilliseconds } from '../../../caching.mjs';
import { METRIC_LABEL } from '../../../constants.mjs';
import { regexifyLabelValuesQueryString } from '../../parsingUtils.mjs';
import { formatPrometheusLabelFilters } from '../MetricCombobox.mjs';
import { AdditionalSettings } from './AdditionalSettings.mjs';
import { FeedbackLink } from './FeedbackLink.mjs';
import { ResultsTable } from './ResultsTable.mjs';
import { setMetrics, promTypes, tracking, placeholders, displayedMetrics, calculatePageList, calculateResultsPerPage } from './state/helpers.mjs';
import { stateSlice, initialState, DEFAULT_RESULTS_PER_PAGE, MAXIMUM_RESULTS_PER_PAGE } from './state/state.mjs';
import { getStyles } from './styles.mjs';
import { debouncedFuzzySearch } from './uFuzzy.mjs';

"use strict";
const MetricsModal = (props) => {
  var _a;
  const { datasource, isOpen, onClose, onChange, query, initialMetrics, timeRange } = props;
  const [state, dispatch] = useReducer(stateSlice.reducer, initialState(query));
  const theme = useTheme2();
  const styles = getStyles(theme, state.disableTextWrap);
  const updateMetricsMetadata = useCallback(async () => {
    dispatch(setIsLoading(true));
    const metrics = typeof initialMetrics === "function" ? await initialMetrics() : initialMetrics;
    const data = await setMetrics(datasource, query, metrics);
    dispatch(
      buildMetrics({
        isLoading: false,
        hasMetadata: data.hasMetadata,
        metrics: data.metrics,
        metaHaystackDictionary: data.metaHaystackDictionary,
        nameHaystackDictionary: data.nameHaystackDictionary,
        totalMetricCount: data.metrics.length,
        filteredMetricCount: data.metrics.length
      })
    );
  }, [query, datasource, initialMetrics]);
  useEffect(() => {
    updateMetricsMetadata();
  }, [updateMetricsMetadata]);
  const typeOptions = promTypes.map((t2) => {
    return {
      value: t2.value,
      label: t2.value,
      description: t2.description
    };
  });
  const debouncedBackendSearch = useMemo(
    () => debounce(async (metricText) => {
      dispatch(setIsLoading(true));
      const queryString = regexifyLabelValuesQueryString(metricText);
      const filterArray = query.labels ? formatPrometheusLabelFilters(query.labels) : [];
      const match = `{__name__=~".*${queryString}"${filterArray ? filterArray.join("") : ""}}`;
      const results = await datasource.languageProvider.queryLabelValues(timeRange, METRIC_LABEL, match);
      const resultsOptions = results.map((result) => ({
        value: result
      }));
      dispatch(
        filterMetricsBackend({
          metrics: resultsOptions,
          filteredMetricCount: resultsOptions.length,
          isLoading: false
        })
      );
    }, getDebounceTimeInMilliseconds(datasource.cacheLevel)),
    [datasource.cacheLevel, datasource.languageProvider, query.labels, timeRange]
  );
  function fuzzyNameDispatch(haystackData) {
    dispatch(setNameHaystack(haystackData));
  }
  function fuzzyMetaDispatch(haystackData) {
    dispatch(setMetaHaystack(haystackData));
  }
  function searchCallback(query2, fullMetaSearchVal) {
    if (state.useBackend && query2 === "") {
      updateMetricsMetadata();
    } else if (state.useBackend) {
      debouncedBackendSearch(query2);
    } else {
      if (fullMetaSearchVal) {
        debouncedFuzzySearch(Object.keys(state.metaHaystackDictionary), query2, fuzzyMetaDispatch);
      } else {
        debouncedFuzzySearch(Object.keys(state.nameHaystackDictionary), query2, fuzzyNameDispatch);
      }
    }
  }
  const additionalSettings = /* @__PURE__ */ jsx(
    AdditionalSettings,
    {
      state,
      onChangeFullMetaSearch: () => {
        const newVal = !state.fullMetaSearch;
        dispatch(setFullMetaSearch(newVal));
        onChange({ ...query, fullMetaSearch: newVal });
        searchCallback(state.fuzzySearchQuery, newVal);
      },
      onChangeIncludeNullMetadata: () => {
        dispatch(setIncludeNullMetadata(!state.includeNullMetadata));
        onChange({ ...query, includeNullMetadata: !state.includeNullMetadata });
      },
      onChangeDisableTextWrap: () => {
        dispatch(setDisableTextWrap());
        onChange({ ...query, disableTextWrap: !state.disableTextWrap });
        tracking("grafana_prom_metric_encycopedia_disable_text_wrap_interaction", state, "");
      },
      onChangeUseBackend: () => {
        const newVal = !state.useBackend;
        dispatch(setUseBackend(newVal));
        onChange({ ...query, useBackend: newVal });
        if (newVal === false) {
          updateMetricsMetadata();
        } else {
          if (state.fuzzySearchQuery !== "") {
            debouncedBackendSearch(state.fuzzySearchQuery);
          }
        }
      }
    }
  );
  return /* @__PURE__ */ jsxs(
    Modal,
    {
      "data-testid": metricsModaltestIds.metricModal,
      isOpen,
      title: t("grafana-prometheus.querybuilder.metrics-modal.title-metrics-explorer", "Metrics explorer"),
      onDismiss: onClose,
      "aria-label": t("grafana-prometheus.querybuilder.metrics-modal.aria-label-browse-metrics", "Browse metrics"),
      className: styles.modal,
      children: [
        /* @__PURE__ */ jsx(FeedbackLink, { feedbackUrl: "https://forms.gle/DEMAJHoAMpe3e54CA" }),
        /* @__PURE__ */ jsxs(
          "div",
          {
            className: styles.inputWrapper,
            "data-testid": selectors.components.DataSource.Prometheus.queryEditor.builder.metricsExplorer,
            children: [
              /* @__PURE__ */ jsx("div", { className: cx(styles.inputItem, styles.inputItemFirst), children: /* @__PURE__ */ jsx(
                Input,
                {
                  autoFocus: true,
                  "data-testid": metricsModaltestIds.searchMetric,
                  placeholder: placeholders.browse,
                  value: state.fuzzySearchQuery,
                  onInput: (e) => {
                    var _a2;
                    const value = (_a2 = e.currentTarget.value) != null ? _a2 : "";
                    dispatch(setFuzzySearchQuery(value));
                    searchCallback(value, state.fullMetaSearch);
                  }
                }
              ) }),
              state.hasMetadata && /* @__PURE__ */ jsx("div", { className: styles.inputItem, children: /* @__PURE__ */ jsx(
                MultiSelect,
                {
                  "data-testid": metricsModaltestIds.selectType,
                  inputId: "my-select",
                  options: typeOptions,
                  value: state.selectedTypes,
                  placeholder: placeholders.type,
                  onChange: (v) => dispatch(setSelectedTypes(v))
                }
              ) }),
              /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(Spinner, { className: `${styles.loadingSpinner} ${state.isLoading ? styles.visible : ""}` }) }),
              /* @__PURE__ */ jsx("div", { className: styles.inputItem, children: /* @__PURE__ */ jsx(
                Toggletip,
                {
                  "aria-label": t(
                    "grafana-prometheus.querybuilder.metrics-modal.aria-label-additional-settings",
                    "Additional settings"
                  ),
                  content: additionalSettings,
                  placement: "bottom-end",
                  closeButton: false,
                  children: /* @__PURE__ */ jsxs(ButtonGroup, { className: styles.settingsBtn, children: [
                    /* @__PURE__ */ jsx(
                      Button,
                      {
                        variant: "secondary",
                        size: "md",
                        onClick: () => dispatch(showAdditionalSettings()),
                        "data-testid": metricsModaltestIds.showAdditionalSettings,
                        className: styles.noBorder,
                        children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.metrics-modal.additional-settings", children: "Additional Settings" })
                      }
                    ),
                    /* @__PURE__ */ jsx(
                      Button,
                      {
                        className: styles.noBorder,
                        variant: "secondary",
                        icon: state.showAdditionalSettings ? "angle-up" : "angle-down"
                      }
                    )
                  ] })
                }
              ) })
            ]
          }
        ),
        /* @__PURE__ */ jsxs("div", { className: styles.resultsData, children: [
          query.metric && /* @__PURE__ */ jsx("i", { className: styles.currentlySelected, children: /* @__PURE__ */ jsxs(
            Trans,
            {
              i18nKey: "grafana-prometheus.querybuilder.metrics-modal.currently-selected",
              values: { selected: query.metric },
              children: [
                "Currently selected: ",
                "{{selected}}"
              ]
            }
          ) }),
          query.labels.length > 0 && /* @__PURE__ */ jsxs("div", { className: styles.resultsDataFiltered, children: [
            /* @__PURE__ */ jsx(Icon, { name: "info-circle", size: "sm" }),
            /* @__PURE__ */ jsxs("div", { className: styles.resultsDataFilteredText, children: [
              "\xA0",
              /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.metrics-modal.metrics-pre-filtered", children: "These metrics have been pre-filtered by labels chosen in the label filters." })
            ] })
          ] })
        ] }),
        /* @__PURE__ */ jsx("div", { className: styles.results, children: state.metrics && /* @__PURE__ */ jsx(
          ResultsTable,
          {
            metrics: displayedMetrics(state, dispatch),
            onChange,
            onClose,
            query,
            state,
            disableTextWrap: state.disableTextWrap
          }
        ) }),
        /* @__PURE__ */ jsxs("div", { className: styles.resultsFooter, children: [
          /* @__PURE__ */ jsx("div", { className: styles.resultsAmount, children: /* @__PURE__ */ jsxs(
            Trans,
            {
              i18nKey: "grafana-prometheus.querybuilder.metrics-modal.results-amount",
              values: { num: state.filteredMetricCount },
              count: state.totalMetricCount,
              children: [
                "Showing ",
                "{{num}}",
                " of ",
                "{{count}}",
                " results"
              ]
            }
          ) }),
          /* @__PURE__ */ jsx(
            Pagination,
            {
              currentPage: (_a = state.pageNum) != null ? _a : 1,
              numberOfPages: calculatePageList(state).length,
              onNavigate: (val) => {
                const page = val != null ? val : 1;
                dispatch(setPageNum(page));
              }
            }
          ),
          /* @__PURE__ */ jsxs("div", { className: styles.resultsPerPageWrapper, children: [
            /* @__PURE__ */ jsx("p", { className: styles.resultsPerPageLabel, children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.metrics-modal.results-per-page", children: "Results per page" }) }),
            /* @__PURE__ */ jsx(
              Input,
              {
                "data-testid": metricsModaltestIds.resultsPerPage,
                value: calculateResultsPerPage(state.resultsPerPage, DEFAULT_RESULTS_PER_PAGE, MAXIMUM_RESULTS_PER_PAGE),
                placeholder: t(
                  "grafana-prometheus.querybuilder.metrics-modal.placeholder-results-per-page",
                  "results per page"
                ),
                width: 10,
                title: "The maximum results per page is " + MAXIMUM_RESULTS_PER_PAGE,
                type: "number",
                onInput: (e) => {
                  const value = +e.currentTarget.value;
                  if (isNaN(value) || value >= MAXIMUM_RESULTS_PER_PAGE) {
                    return;
                  }
                  dispatch(setResultsPerPage(value));
                }
              }
            )
          ] })
        ] })
      ]
    }
  );
};
const metricsModaltestIds = {
  metricModal: "metric-modal",
  searchMetric: "search-metric",
  searchWithMetadata: "search-with-metadata",
  selectType: "select-type",
  metricCard: "metric-card",
  useMetric: "use-metric",
  searchPage: "search-page",
  resultsPerPage: "results-per-page",
  setUseBackend: "set-use-backend",
  showAdditionalSettings: "show-additional-settings"
};
const {
  setIsLoading,
  buildMetrics,
  filterMetricsBackend,
  setResultsPerPage,
  setPageNum,
  setFuzzySearchQuery,
  setNameHaystack,
  setMetaHaystack,
  setFullMetaSearch,
  setIncludeNullMetadata,
  setSelectedTypes,
  setUseBackend,
  setDisableTextWrap,
  showAdditionalSettings,
  setFilteredMetricCount
} = stateSlice.actions;

export { MetricsModal, buildMetrics, filterMetricsBackend, metricsModaltestIds, setDisableTextWrap, setFilteredMetricCount, setFullMetaSearch, setFuzzySearchQuery, setIncludeNullMetadata, setIsLoading, setMetaHaystack, setNameHaystack, setPageNum, setResultsPerPage, setSelectedTypes, setUseBackend, showAdditionalSettings };
//# sourceMappingURL=MetricsModal.mjs.map
