import { jsx } from 'react/jsx-runtime';
import { isFunction } from 'lodash';
import { useState } from 'react';
import { ThresholdsMode, getFieldConfigWithMinMax, VizOrientation } from '@grafana/data';
import { BarGaugeDisplayMode, TableCellDisplayMode, BarGaugeValueMode } from '@grafana/schema';
import { BarGauge } from '../../BarGauge/BarGauge.mjs';
import { renderSingleLink, DataLinksActionsTooltip } from '../DataLinksActionsTooltip.mjs';
import { getCellOptions, getAlignmentFactor, getDataLinksActionsTooltipUtils } from '../utils.mjs';

"use strict";
const defaultScale = {
  mode: ThresholdsMode.Absolute,
  steps: [
    {
      color: "blue",
      value: -Infinity
    },
    {
      color: "green",
      value: 20
    }
  ]
};
const BarGaugeCell = (props) => {
  var _a;
  const { field, innerWidth, tableStyles, cell, cellProps, row } = props;
  const displayValue = field.display(cell.value);
  const cellOptions = getCellOptions(field);
  let config = getFieldConfigWithMinMax(field, false);
  if (!config.thresholds) {
    config = {
      ...config,
      thresholds: defaultScale
    };
  }
  let barGaugeMode = BarGaugeDisplayMode.Gradient;
  let valueDisplayMode = void 0;
  if (cellOptions.type === TableCellDisplayMode.Gauge) {
    barGaugeMode = (_a = cellOptions.mode) != null ? _a : BarGaugeDisplayMode.Gradient;
    valueDisplayMode = cellOptions.valueDisplayMode !== void 0 ? cellOptions.valueDisplayMode : BarGaugeValueMode.Text;
  }
  const getLinks = () => {
    if (!isFunction(field.getLinks)) {
      return [];
    }
    return field.getLinks({ valueRowIndex: row.index });
  };
  const alignmentFactors = getAlignmentFactor(field, displayValue, cell.row.index);
  const renderComponent = () => {
    return /* @__PURE__ */ jsx(
      BarGauge,
      {
        width: innerWidth,
        height: tableStyles.cellHeightInner,
        field: config,
        display: field.display,
        text: { valueSize: 14 },
        value: displayValue,
        orientation: VizOrientation.Horizontal,
        theme: tableStyles.theme,
        alignmentFactors,
        itemSpacing: 1,
        lcdCellWidth: 8,
        displayMode: barGaugeMode,
        valueDisplayMode
      }
    );
  };
  const [tooltipCoords, setTooltipCoords] = useState();
  const { shouldShowLink, hasMultipleLinksOrActions } = getDataLinksActionsTooltipUtils(getLinks());
  const shouldShowTooltip = hasMultipleLinksOrActions && tooltipCoords !== void 0;
  const links = getLinks();
  return (
    // eslint-disable-next-line jsx-a11y/click-events-have-key-events, jsx-a11y/no-static-element-interactions
    /* @__PURE__ */ jsx(
      "div",
      {
        ...cellProps,
        className: tableStyles.cellContainer,
        style: { ...cellProps.style, cursor: hasMultipleLinksOrActions ? "context-menu" : "auto" },
        onClick: ({ clientX, clientY }) => {
          setTooltipCoords({ clientX, clientY });
        },
        children: shouldShowLink ? renderSingleLink(links[0], renderComponent()) : shouldShowTooltip ? /* @__PURE__ */ jsx(
          DataLinksActionsTooltip,
          {
            links,
            value: renderComponent(),
            coords: tooltipCoords,
            onTooltipClose: () => setTooltipCoords(void 0)
          }
        ) : renderComponent()
      }
    )
  );
};

export { BarGaugeCell };
//# sourceMappingURL=BarGaugeCell.mjs.map
