import { jsx, Fragment } from 'react/jsx-runtime';
import { useRef, useState, useLayoutEffect, useMemo, createElement } from 'react';
import { useMountedState } from 'react-use';
import { Marker } from './Marker.mjs';
import { XYCanvas } from './XYCanvas.mjs';

"use strict";
function EventsCanvas({ id, events, renderEventMarker, mapEventToXYCoords, config }) {
  const plotInstance = useRef();
  const [renderToken, setRenderToken] = useState(0);
  const isMounted = useMountedState();
  useLayoutEffect(() => {
    config.addHook("init", (u) => {
      plotInstance.current = u;
    });
    config.addHook("draw", () => {
      if (!isMounted()) {
        return;
      }
      setRenderToken((s) => s + 1);
    });
  }, [config, setRenderToken]);
  const eventMarkers = useMemo(() => {
    const markers = [];
    if (!plotInstance.current || events.length === 0) {
      return markers;
    }
    for (let i = 0; i < events.length; i++) {
      const frame = events[i];
      for (let j = 0; j < frame.length; j++) {
        const coords = mapEventToXYCoords(frame, { fieldIndex: j, frameIndex: i });
        if (!coords) {
          continue;
        }
        markers.push(
          /* @__PURE__ */ createElement(Marker, { ...coords, key: `${id}-marker-${i}-${j}` }, renderEventMarker(frame, { fieldIndex: j, frameIndex: i }))
        );
      }
    }
    return /* @__PURE__ */ jsx(Fragment, { children: markers });
  }, [events, renderEventMarker, renderToken]);
  if (!plotInstance.current) {
    return null;
  }
  return /* @__PURE__ */ jsx(
    XYCanvas,
    {
      left: plotInstance.current.bbox.left / window.devicePixelRatio,
      top: plotInstance.current.bbox.top / window.devicePixelRatio,
      children: eventMarkers
    }
  );
}

export { EventsCanvas };
//# sourceMappingURL=EventsCanvas.mjs.map
