import { jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import * as React from 'react';
import { deprecationWarning, colorManipulator } from '@grafana/data';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import { getFocusStyles, getMouseFocusStyles } from '../../themes/mixins.mjs';
import { IconRenderer } from '../Button/Button.mjs';
import { getSvgSize } from '../Icon/utils.mjs';
import { Tooltip } from '../Tooltip/Tooltip.mjs';

"use strict";
const IconButton = React.forwardRef((props, ref) => {
  const { size = "md", variant = "secondary" } = props;
  let limitedIconSize;
  if (size === "xxl" || size === "xxxl") {
    deprecationWarning("IconButton", 'size="xxl" and size="xxxl"', 'size="xl"');
    limitedIconSize = "xl";
  } else {
    limitedIconSize = size;
  }
  const styles = useStyles2(getStyles, limitedIconSize, variant);
  let ariaLabel;
  let buttonRef;
  if ("tooltip" in props) {
    const { tooltip } = props;
    ariaLabel = typeof tooltip === "string" ? tooltip : void 0;
  } else if ("ariaLabel" in props || "aria-label" in props) {
    const { ariaLabel: deprecatedAriaLabel, ["aria-label"]: ariaLabelProp } = props;
    ariaLabel = ariaLabelProp || deprecatedAriaLabel;
    buttonRef = ref;
  }
  if ("tooltip" in props) {
    const { name, iconType, className, tooltip, tooltipPlacement, ...restProps } = props;
    return /* @__PURE__ */ jsx(Tooltip, { ref, content: tooltip, placement: tooltipPlacement, children: /* @__PURE__ */ jsx(
      "button",
      {
        ...restProps,
        ref: buttonRef,
        "aria-label": ariaLabel,
        className: cx(styles.button, className),
        type: "button",
        children: /* @__PURE__ */ jsx(IconRenderer, { icon: name, size: limitedIconSize, className: styles.icon, iconType })
      }
    ) });
  } else {
    const { name, iconType, className, ...restProps } = props;
    return /* @__PURE__ */ jsx(
      "button",
      {
        ...restProps,
        ref: buttonRef,
        "aria-label": ariaLabel,
        className: cx(styles.button, className),
        type: "button",
        children: /* @__PURE__ */ jsx(IconRenderer, { icon: name, size: limitedIconSize, className: styles.icon, iconType })
      }
    );
  }
});
IconButton.displayName = "IconButton";
const getStyles = (theme, size, variant) => {
  const hoverSize = getSvgSize(size) + theme.spacing.gridSize;
  let iconColor = theme.colors.text.primary;
  if (variant === "primary") {
    iconColor = theme.colors.primary.text;
  } else if (variant === "destructive") {
    iconColor = theme.colors.error.text;
  }
  return {
    button: css({
      zIndex: 0,
      position: "relative",
      margin: `0 ${theme.spacing.x0_5} 0 0`,
      boxShadow: "none",
      border: "none",
      display: "inline-flex",
      background: "transparent",
      justifyContent: "center",
      alignItems: "center",
      padding: 0,
      color: iconColor,
      "&[disabled], &:disabled": {
        cursor: "not-allowed",
        color: theme.colors.action.disabledText,
        opacity: 0.65
      },
      "&:before": {
        zIndex: -1,
        position: "absolute",
        opacity: 0,
        width: `${hoverSize}px`,
        height: `${hoverSize}px`,
        borderRadius: theme.shape.radius.default,
        content: '""',
        [theme.transitions.handleMotion("no-preference", "reduce")]: {
          transitionDuration: "0.2s",
          transitionTimingFunction: "cubic-bezier(0.4, 0, 0.2, 1)",
          transitionProperty: "opacity"
        }
      },
      "&:focus, &:focus-visible": getFocusStyles(theme),
      "&:focus:not(:focus-visible)": getMouseFocusStyles(theme),
      "&:hover": {
        "&:before": {
          backgroundColor: variant === "secondary" ? theme.colors.action.hover : colorManipulator.alpha(iconColor, 0.12),
          opacity: 1
        }
      }
    }),
    icon: css({
      verticalAlign: "baseline"
    })
  };
};

export { IconButton };
//# sourceMappingURL=IconButton.mjs.map
