import { once } from 'lodash';
import Prism from 'prismjs';
import { LanguageProvider, getDefaultTimeRange, scopeFilterOperatorMap, AbstractLabelOperator } from '@grafana/data';
import { getDefaultCacheHeaders, buildCacheHeaders, getDaysToCacheMetadata } from './caching.mjs';
import { REMOVE_SERIES_LIMIT, DEFAULT_SERIES_LIMIT, EMPTY_SELECTOR } from './constants.mjs';
import { processHistogramMetrics, processLabels, fixSummariesMetadata, toPromLikeQuery, extractLabelMatchers } from './language_utils.mjs';
import { promqlGrammar } from './promql.mjs';
import { buildVisualQueryFromString } from './querybuilder/parsing.mjs';
import { LabelsApiClient, SeriesApiClient } from './resource_clients.mjs';
import { escapeForUtf8Support, isValidLegacyName } from './utf8_support.mjs';

"use strict";
const DEFAULT_KEYS = ["job", "instance"];
const API_V1 = {
  METADATA: "/api/v1/metadata",
  SERIES: "/api/v1/series",
  LABELS: "/api/v1/labels",
  LABELS_VALUES: (labelKey) => `/api/v1/label/${labelKey}/values`
};
class PromQlLanguageProvider extends LanguageProvider {
  constructor(datasource, initialValues) {
    super();
    this.labelKeys = [];
    this.request = async (url, params = {}, options) => {
      try {
        const res = await this.datasource.metadataRequest(url, params, options);
        return res.data.data;
      } catch (error) {
        if (!isCancelledError(error)) {
          console.error(error);
        }
      }
      return void 0;
    };
    /**
     * Overridden by PrometheusLanguageProvider
     */
    this.start = async (timeRange = getDefaultTimeRange()) => {
      if (this.datasource.lookupsDisabled) {
        return [];
      }
      this.metrics = await this.fetchLabelValues(timeRange, "__name__") || [];
      this.histogramMetrics = processHistogramMetrics(this.metrics).sort();
      return Promise.all([this.loadMetricsMetadata(), this.fetchLabels(timeRange)]);
    };
    this.fetchLabelValues = async (range, key, limit) => {
      const params = { ...this.datasource.getAdjustedInterval(range), ...limit ? { limit } : {} };
      const interpolatedName = this.datasource.interpolateString(key);
      const interpolatedAndEscapedName = escapeForUtf8Support(removeQuotesIfExist(interpolatedName));
      const value = await this.request(
        API_V1.LABELS_VALUES(interpolatedAndEscapedName),
        params,
        getDefaultCacheHeaders(this.datasource.cacheLevel)
      );
      return value != null ? value : [];
    };
    /**
     * Fetches all label keys
     */
    this.fetchLabels = async (timeRange, queries, limit) => {
      let url = API_V1.LABELS;
      const timeParams = this.datasource.getAdjustedInterval(timeRange);
      this.labelFetchTs = Date.now().valueOf();
      const searchParams = new URLSearchParams({ ...timeParams, ...limit ? { limit } : {} });
      queries == null ? void 0 : queries.forEach((q) => {
        const visualQuery = buildVisualQueryFromString(q.expr);
        if (visualQuery.query.metric !== "") {
          const isUtf8Metric = !isValidLegacyName(visualQuery.query.metric);
          searchParams.append("match[]", isUtf8Metric ? `{"${visualQuery.query.metric}"}` : visualQuery.query.metric);
          if (visualQuery.query.binaryQueries) {
            visualQuery.query.binaryQueries.forEach((bq) => {
              searchParams.append("match[]", isUtf8Metric ? `{"${bq.query.metric}"}` : bq.query.metric);
            });
          }
        }
      });
      if (this.datasource.httpMethod === "GET") {
        url += `?${searchParams.toString()}`;
      }
      const res = await this.request(url, searchParams, getDefaultCacheHeaders(this.datasource.cacheLevel));
      if (Array.isArray(res)) {
        this.labelKeys = res.slice().sort();
        return [...this.labelKeys];
      }
      return [];
    };
    /**
     * Gets series values
     * Function to replace old getSeries calls in a way that will provide faster endpoints
     * for new prometheus instances, while maintaining backward compatability
     */
    this.getSeriesValues = async (timeRange, labelName, selector) => {
      var _a;
      if (!this.datasource.hasLabelsMatchAPISupport()) {
        const data = await this.getSeries(timeRange, selector);
        return (_a = data[removeQuotesIfExist(labelName)]) != null ? _a : [];
      }
      return await this.fetchSeriesValuesWithMatch(timeRange, labelName, selector);
    };
    /**
     * Fetches all values for a label, with optional match[]
     */
    this.fetchSeriesValuesWithMatch = async (timeRange, name, match, requestId, withLimit) => {
      const interpolatedName = name ? this.datasource.interpolateString(name) : null;
      const interpolatedMatch = match ? this.datasource.interpolateString(match) : null;
      const range = this.datasource.getAdjustedInterval(timeRange);
      const urlParams = {
        ...range,
        ...interpolatedMatch && { "match[]": interpolatedMatch },
        ...withLimit ? { limit: withLimit } : {}
      };
      let requestOptions = {
        ...getDefaultCacheHeaders(this.datasource.cacheLevel),
        ...requestId && { requestId }
      };
      if (!Object.keys(requestOptions).length) {
        requestOptions = void 0;
      }
      const interpolatedAndEscapedName = escapeForUtf8Support(removeQuotesIfExist(interpolatedName != null ? interpolatedName : ""));
      const value = await this.request(API_V1.LABELS_VALUES(interpolatedAndEscapedName), urlParams, requestOptions);
      return value != null ? value : [];
    };
    /**
     * Gets series labels
     * Function to replace old getSeries calls in a way that will provide faster endpoints for new prometheus instances,
     * while maintaining backward compatability. The old API call got the labels and the values in a single query,
     * but with the new query we need two calls, one to get the labels, and another to get the values.
     */
    this.getSeriesLabels = async (timeRange, selector, otherLabels) => {
      let possibleLabelNames, data;
      if (!this.datasource.hasLabelsMatchAPISupport()) {
        data = await this.getSeries(timeRange, selector);
        possibleLabelNames = Object.keys(data);
      } else {
        otherLabels.push({ name: "__name__", value: "", op: "!=" });
        data = await this.fetchSeriesLabelsMatch(timeRange, selector);
        possibleLabelNames = Object.keys(data);
      }
      const usedLabelNames = new Set(otherLabels.map((l) => l.name));
      return possibleLabelNames.filter((l) => !usedLabelNames.has(l));
    };
    /**
     * Fetch labels using the best endpoint that datasource supports.
     * This is cached by its args but also by the global timeRange currently selected as they can change over requested time.
     */
    this.fetchLabelsWithMatch = async (timeRange, name, withName, withLimit) => {
      if (this.datasource.hasLabelsMatchAPISupport()) {
        return this.fetchSeriesLabelsMatch(timeRange, name, withLimit);
      } else {
        return this.fetchSeriesLabels(timeRange, name, withName, REMOVE_SERIES_LIMIT);
      }
    };
    /**
     * Fetch labels for a series using /series endpoint. This is cached by its args but also by the global timeRange currently selected as
     * they can change over requested time.
     */
    this.fetchSeriesLabels = async (timeRange, name, withName, withLimit) => {
      const interpolatedName = this.datasource.interpolateString(name);
      const range = this.datasource.getAdjustedInterval(timeRange);
      let urlParams = {
        ...range,
        "match[]": interpolatedName,
        ...withLimit !== "none" ? { limit: withLimit != null ? withLimit : DEFAULT_SERIES_LIMIT } : {}
      };
      const data = await this.request(API_V1.SERIES, urlParams, getDefaultCacheHeaders(this.datasource.cacheLevel));
      const { values } = processLabels(data, withName);
      return values;
    };
    /**
     * Fetch labels for a series using /labels endpoint.  This is cached by its args but also by the global timeRange currently selected as
     * they can change over requested time.
     */
    this.fetchSeriesLabelsMatch = async (timeRange, name, withLimit) => {
      const interpolatedName = this.datasource.interpolateString(name);
      const range = this.datasource.getAdjustedInterval(timeRange);
      const urlParams = {
        ...range,
        "match[]": interpolatedName,
        ...withLimit ? { limit: withLimit } : {}
      };
      const data = await this.request(
        API_V1.LABELS,
        urlParams,
        getDefaultCacheHeaders(this.datasource.cacheLevel)
      );
      return data.reduce((ac, a) => ({ ...ac, [a]: "" }), {});
    };
    /**
     * Fetch series for a selector. Use this for raw results. Use fetchSeriesLabels() to get labels.
     */
    this.fetchSeries = async (timeRange, match) => {
      const range = this.datasource.getTimeRangeParams(timeRange);
      const params = { ...range, "match[]": match };
      return await this.request(API_V1.SERIES, params, getDefaultCacheHeaders(this.datasource.cacheLevel));
    };
    /**
     * Fetch this only one as we assume this won't change over time. This is cached differently from fetchSeriesLabels
     * because we can cache more aggressively here and also we do not want to invalidate this cache the same way as in
     * fetchSeriesLabels.
     */
    this.fetchDefaultSeries = once(async (timeRange) => {
      const values = await Promise.all(DEFAULT_KEYS.map((key) => this.fetchLabelValues(timeRange, key)));
      return DEFAULT_KEYS.reduce((acc, key, i) => ({ ...acc, [key]: values[i] }), {});
    });
    /**
     * Fetch labels or values for a label based on the queries, scopes, filters and time range
     */
    this.fetchSuggestions = async (timeRange, queries, scopes, adhocFilters, labelName, limit, requestId) => {
      var _a;
      if (!timeRange) {
        timeRange = getDefaultTimeRange();
      }
      const url = "/suggestions";
      const timeParams = this.datasource.getAdjustedInterval(timeRange);
      const value = await this.request(
        url,
        {
          labelName,
          queries: queries == null ? void 0 : queries.map(
            (q) => this.datasource.interpolateString(q.expr, {
              ...this.datasource.getIntervalVars(),
              ...this.datasource.getRangeScopedVars(timeRange)
            })
          ),
          scopes: scopes == null ? void 0 : scopes.reduce((acc, scope) => {
            acc.push(...scope.spec.filters);
            return acc;
          }, []),
          adhocFilters: adhocFilters == null ? void 0 : adhocFilters.map((filter) => ({
            key: filter.key,
            operator: scopeFilterOperatorMap[filter.operator],
            value: filter.value,
            values: filter.values
          })),
          limit,
          ...timeParams
        },
        {
          ...requestId && { requestId },
          headers: {
            ...(_a = getDefaultCacheHeaders(this.datasource.cacheLevel)) == null ? void 0 : _a.headers,
            "Content-Type": "application/json"
          },
          method: "POST"
        }
      );
      return value != null ? value : [];
    };
    this.datasource = datasource;
    this.histogramMetrics = [];
    this.metrics = [];
    Object.assign(this, initialValues);
  }
  async loadMetricsMetadata() {
    const secondsInDay = 86400;
    const headers = buildCacheHeaders(getDaysToCacheMetadata(this.datasource.cacheLevel) * secondsInDay);
    this.metricsMetadata = fixSummariesMetadata(
      await this.request(
        API_V1.METADATA,
        {},
        {
          showErrorAlert: false,
          ...headers
        }
      )
    );
  }
  getLabelKeys() {
    return this.labelKeys;
  }
  async getSeries(timeRange, selector, withName) {
    if (this.datasource.lookupsDisabled) {
      return {};
    }
    try {
      if (selector === EMPTY_SELECTOR) {
        return await this.fetchDefaultSeries(timeRange);
      } else {
        return await this.fetchSeriesLabels(timeRange, selector, withName, REMOVE_SERIES_LIMIT);
      }
    } catch (error) {
      console.error(error);
      return {};
    }
  }
  async getLabelValues(range, key) {
    return await this.fetchLabelValues(range, key);
  }
}
class PrometheusLanguageProvider extends PromQlLanguageProvider {
  constructor(datasource) {
    super(datasource);
    /**
     * Same start logic but it uses resource clients. Backward compatibility it calls _backwardCompatibleStart.
     * Some places still relies on deprecated fields. Until we replace them we need _backwardCompatibleStart method
     */
    this.start = async (timeRange = getDefaultTimeRange()) => {
      if (this.datasource.lookupsDisabled) {
        return [];
      }
      await Promise.all([this.resourceClient.start(timeRange), this.queryMetricsMetadata(this.datasource.seriesLimit)]);
      return this._backwardCompatibleStart();
    };
    /**
     * This private method exists to make sure the old class will be functional until we remove it.
     * When we remove old class (PromQlLanguageProvider) we should remove this method too.
     */
    this._backwardCompatibleStart = async () => {
      this.metricsMetadata = this.retrieveMetricsMetadata();
      this.metrics = this.retrieveMetrics();
      this.histogramMetrics = this.retrieveHistogramMetrics();
      this.labelKeys = this.retrieveLabelKeys();
      return [];
    };
    /**
     * Fetches metadata for metrics from Prometheus.
     * Sets cache headers based on the configured metadata cache duration.
     *
     * @returns {Promise<PromMetricsMetadata>} Promise that resolves when metadata has been fetched
     */
    this._queryMetadata = async (limit) => {
      const secondsInDay = 86400;
      const headers = buildCacheHeaders(getDaysToCacheMetadata(this.datasource.cacheLevel) * secondsInDay);
      const metadata = await this.request(
        API_V1.METADATA,
        { limit: limit != null ? limit : this.datasource.seriesLimit },
        {
          showErrorAlert: false,
          ...headers
        }
      );
      return fixSummariesMetadata(metadata);
    };
    /**
     * Retrieves the cached Prometheus metrics metadata.
     * This metadata includes type information (counter, gauge, etc.) and help text for metrics.
     *
     * @returns {PromMetricsMetadata} Cached metadata or empty object if not yet fetched
     */
    this.retrieveMetricsMetadata = () => {
      var _a;
      return (_a = this._metricsMetadata) != null ? _a : {};
    };
    /**
     * Retrieves the list of histogram metrics from the current resource client.
     * Histogram metrics are identified by the '_bucket' suffix and are used for percentile calculations.
     *
     * @returns {string[]} Array of histogram metric names
     */
    this.retrieveHistogramMetrics = () => {
      var _a;
      return (_a = this.resourceClient) == null ? void 0 : _a.histogramMetrics;
    };
    /**
     * Retrieves the complete list of available metrics from the current resource client.
     * This includes all metric names regardless of their type (counter, gauge, histogram).
     *
     * @returns {string[]} Array of all metric names
     */
    this.retrieveMetrics = () => {
      var _a;
      return (_a = this.resourceClient) == null ? void 0 : _a.metrics;
    };
    /**
     * Retrieves the list of available label keys from the current resource client.
     * Label keys are the names of labels that can be used to filter and group metrics.
     *
     * @returns {string[]} Array of label key names
     */
    this.retrieveLabelKeys = () => {
      var _a;
      return (_a = this.resourceClient) == null ? void 0 : _a.labelKeys;
    };
    /**
     * Fetches fresh metrics metadata from Prometheus and updates the cache.
     * This includes querying for metric types, help text, and unit information.
     * If the fetch fails, the cache is set to an empty object to prevent stale data.
     *
     * @returns {Promise<PromMetricsMetadata>} Promise that resolves to the fetched metadata
     */
    this.queryMetricsMetadata = async (limit) => {
      var _a;
      try {
        this._metricsMetadata = (_a = await this._queryMetadata(limit)) != null ? _a : {};
      } catch (error) {
        this._metricsMetadata = {};
      }
      return this._metricsMetadata;
    };
    /**
     * Fetches all available label keys that match the specified criteria.
     *
     * This method queries Prometheus for label keys within the specified time range.
     * The results can be filtered using the match parameter and limited in size.
     * Uses either the labels API (Prometheus v2.6+) or series API based on version.
     *
     * @param {TimeRange} timeRange - Time range to search for label keys
     * @param {string} [match] - Optional PromQL selector to filter label keys (e.g., '{job="grafana"}')
     * @param {string} [limit] - Optional maximum number of label keys to return
     * @returns {Promise<string[]>} Array of matching label key names, sorted alphabetically
     */
    this.queryLabelKeys = async (timeRange, match, limit) => {
      const interpolatedMatch = match ? this.datasource.interpolateString(match) : match;
      return await this.resourceClient.queryLabelKeys(timeRange, interpolatedMatch, limit);
    };
    /**
     * Fetches all values for a specific label key that match the specified criteria.
     *
     * This method queries Prometheus for label values within the specified time range.
     * Results can be filtered using the match parameter to find values in specific contexts.
     * Supports both modern (labels API) and legacy (series API) Prometheus versions.
     *
     * The method automatically handles UTF-8 encoded label keys by properly escaping them
     * before making API requests. This means you can safely pass label keys containing
     * special characters like dots, colons, or Unicode characters (e.g., 'http.status:code',
     * 'μs', 'response.time').
     *
     * @param {TimeRange} timeRange - Time range to search for label values
     * @param {string} labelKey - The label key to fetch values for (e.g., 'job', 'instance', 'http.status:code')
     * @param {string} [match] - Optional PromQL selector to filter values (e.g., '{job="grafana"}')
     * @param {string} [limit] - Optional maximum number of values to return
     * @returns {Promise<string[]>} Array of matching label values, sorted alphabetically
     * @example
     * // Fetch all values for the 'job' label
     * const values = await queryLabelValues(timeRange, 'job');
     * // Fetch 'instance' values only for jobs matching 'grafana'
     * const instances = await queryLabelValues(timeRange, 'instance', '{job="grafana"}');
     * // Fetch values for a label key with special characters
     * const statusCodes = await queryLabelValues(timeRange, 'http.status:code');
     */
    this.queryLabelValues = async (timeRange, labelKey, match, limit) => {
      const interpolatedMatch = match ? this.datasource.interpolateString(match) : match;
      return await this.resourceClient.queryLabelValues(
        timeRange,
        this.datasource.interpolateString(labelKey),
        interpolatedMatch,
        limit
      );
    };
  }
  /**
   * Lazily initializes and returns the appropriate resource client based on Prometheus version.
   *
   * The client selection logic:
   * - For Prometheus v2.6+ with labels API: Uses LabelsApiClient for efficient label-based queries
   * - For older versions: Falls back to SeriesApiClient for backward compatibility
   *
   * The client instance is cached after first initialization to avoid repeated creation.
   *
   * @returns {ResourceApiClient} An instance of either LabelsApiClient or SeriesApiClient
   */
  get resourceClient() {
    if (!this._resourceClient) {
      this._resourceClient = this.datasource.hasLabelsMatchAPISupport() ? new LabelsApiClient(this.request, this.datasource) : new SeriesApiClient(this.request, this.datasource);
    }
    return this._resourceClient;
  }
}
const importFromAbstractQuery = (labelBasedQuery) => {
  return toPromLikeQuery(labelBasedQuery);
};
const exportToAbstractQuery = (query) => {
  const promQuery = query.expr;
  if (!promQuery || promQuery.length === 0) {
    return { refId: query.refId, labelMatchers: [] };
  }
  const tokens = Prism.tokenize(promQuery, promqlGrammar);
  const labelMatchers = extractLabelMatchers(tokens);
  const nameLabelValue = getNameLabelValue(promQuery, tokens);
  if (nameLabelValue && nameLabelValue.length > 0) {
    labelMatchers.push({
      name: "__name__",
      operator: AbstractLabelOperator.Equal,
      value: nameLabelValue
    });
  }
  return {
    refId: query.refId,
    labelMatchers
  };
};
function isCancelledError(error) {
  return typeof error === "object" && error !== null && "cancelled" in error && error.cancelled === true;
}
function removeQuotesIfExist(input) {
  var _a;
  const match = input.match(/^"(.*)"$/);
  return (_a = match == null ? void 0 : match[1]) != null ? _a : input;
}
function getNameLabelValue(promQuery, tokens) {
  let nameLabelValue = "";
  for (const token of tokens) {
    if (typeof token === "string") {
      nameLabelValue = token;
      break;
    }
  }
  return nameLabelValue;
}
const populateMatchParamsFromQueries = (queries) => {
  if (!queries) {
    return [];
  }
  const metrics = (queries != null ? queries : []).reduce((params, query) => {
    const visualQuery = buildVisualQueryFromString(query.expr);
    if (visualQuery.query.metric !== "") {
      params.push(visualQuery.query.metric);
    }
    if (visualQuery.query.binaryQueries) {
      visualQuery.query.binaryQueries.forEach((bq) => {
        if (bq.query.metric !== "") {
          params.push(bq.query.metric);
        }
      });
    }
    return params;
  }, []);
  return metrics.length === 0 ? [] : [`__name__=~"${metrics.join("|")}"`];
};

export { PrometheusLanguageProvider, PromQlLanguageProvider as default, exportToAbstractQuery, importFromAbstractQuery, populateMatchParamsFromQueries, removeQuotesIfExist };
//# sourceMappingURL=language_provider.mjs.map
