/**
 * Ensures a string is compatible with Prometheus' UTF-8 handling rules.
 *
 * Prometheus has specific rules for handling UTF-8 strings in metric names and label values:
 * - Legacy names (matching pattern [a-zA-Z_:][a-zA-Z0-9_:]*) are used as-is
 * - Non-legacy names containing UTF-8 characters must be wrapped in double quotes
 *
 * @param value - The string to make UTF-8 compatible
 * @returns The original string if it's empty or a valid legacy name, otherwise the string wrapped in double quotes
 *
 * @example
 * utf8Support('metric_name') // returns 'metric_name'
 * utf8Support('metric-📈') // returns '"metric-📈"'
 */
export declare const utf8Support: (value: string) => string;
/**
 * Escapes a string to make it compatible with Prometheus UTF-8 support.
 *
 * This function converts non-legacy name characters to an escaped format:
 * - Underscores are doubled as '__'
 * - Valid legacy runes are preserved as-is
 * - Invalid code points are replaced with '_FFFD_'
 * - Other characters are converted to '_HEX_' format where HEX is the hexadecimal code point
 *
 * @param value - The string to escape
 * @returns An escaped string prefixed with 'U__' that is compatible with Prometheus
 *
 * @example
 * escapeForUtf8Support("my lovely_http.status:sum") // returns U__my_20_lovely__http_2e_status:sum
 * escapeForUtf8Support("label with 😱") // returns U__label_20_with_20__1f631_
 */
export declare const escapeForUtf8Support: (value: string) => string;
/**
 * Checks if a string is a valid legacy (the standard) Prometheus metric or label name.
 *
 * Valid legacy (the standard) names match the pattern [a-zA-Z_:][a-zA-Z0-9_:]* which means:
 * - First character must be a letter, underscore, or colon
 * - Remaining characters can only be letters, numbers, underscores, or colons
 *
 * @param name - The string to check
 * @returns true if the string is a valid legacy (the standard) name, false otherwise
 */
export declare const isValidLegacyName: (name: string) => boolean;
/**
 * Wraps each key in a Prometheus filter string with UTF-8 support.
 *
 * This function processes a filter string (e.g. 'metric="value",name=~"pattern"')
 * and applies UTF-8 support to each key while preserving the operators and values.
 * It handles quoted values and comma separators correctly.
 *
 * @param filterStr - The filter string to process
 * @returns A new filter string with UTF-8 support applied to the keys
 */
export declare const wrapUtf8Filters: (filterStr: string) => string;
