import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { useState, useCallback } from 'react';
import { selectors } from '@grafana/e2e-selectors';
import { t, Trans } from '@grafana/i18n';
import { InputGroup, EditorFieldGroup, EditorField } from '@grafana/plugin-ui';
import { Combobox, Button, InlineFieldRow, InlineField } from '@grafana/ui';
import { METRIC_LABEL } from '../../constants.mjs';
import { regexifyLabelValuesQueryString } from '../parsingUtils.mjs';
import { MetricsModal } from './metrics-modal/MetricsModal.mjs';
import { tracking } from './metrics-modal/state/helpers.mjs';

"use strict";
function MetricCombobox({
  datasource,
  query,
  onChange,
  onGetMetrics,
  labelsFilters,
  variableEditor,
  timeRange
}) {
  const [metricsModalOpen, setMetricsModalOpen] = useState(false);
  const getMetricLabels = useCallback(
    async (query2) => {
      const match = formatKeyValueStrings(query2, labelsFilters);
      const results = await datasource.languageProvider.queryLabelValues(timeRange, METRIC_LABEL, match);
      const resultsOptions = results.map((result) => {
        return {
          label: result,
          value: result
        };
      });
      return resultsOptions;
    },
    [datasource.languageProvider, labelsFilters, timeRange]
  );
  const onComboboxChange = useCallback(
    (opt) => {
      var _a;
      onChange({ ...query, metric: (_a = opt == null ? void 0 : opt.value) != null ? _a : "" });
    },
    [onChange, query]
  );
  const loadOptions = useCallback(
    async (input) => {
      const metrics = input.length ? await getMetricLabels(input) : await onGetMetrics();
      return metrics.map((option) => {
        var _a;
        return {
          label: (_a = option.label) != null ? _a : option.value,
          value: option.value
        };
      });
    },
    [getMetricLabels, onGetMetrics]
  );
  const loadMetricsExplorerMetrics = useCallback(async () => {
    const allMetrics = await onGetMetrics();
    const metrics = [];
    for (const metric of allMetrics) {
      if (metric.value) {
        metrics.push(metric.value);
      }
    }
    return metrics;
  }, [onGetMetrics]);
  const asyncSelect = () => {
    return /* @__PURE__ */ jsxs(InputGroup, { children: [
      /* @__PURE__ */ jsx(
        Combobox,
        {
          placeholder: t(
            "grafana-prometheus.querybuilder.metric-combobox.async-select.placeholder-select-metric",
            "Select metric"
          ),
          width: "auto",
          minWidth: 25,
          options: loadOptions,
          value: query.metric,
          onChange: onComboboxChange,
          createCustomValue: true,
          "data-testid": selectors.components.DataSource.Prometheus.queryEditor.builder.metricSelect
        }
      ),
      /* @__PURE__ */ jsx(
        Button,
        {
          tooltip: t(
            "grafana-prometheus.querybuilder.metric-combobox.async-select.tooltip-open-metrics-explorer",
            "Open metrics explorer"
          ),
          "aria-label": t(
            "grafana-prometheus.querybuilder.metric-combobox.async-select.aria-label-open-metrics-explorer",
            "Open metrics explorer"
          ),
          variant: "secondary",
          icon: "book-open",
          onClick: () => {
            tracking("grafana_prometheus_metric_encyclopedia_open", null, "", query);
            setMetricsModalOpen(true);
          }
        }
      )
    ] });
  };
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    !datasource.lookupsDisabled && metricsModalOpen && /* @__PURE__ */ jsx(
      MetricsModal,
      {
        datasource,
        isOpen: metricsModalOpen,
        onClose: () => setMetricsModalOpen(false),
        query,
        onChange,
        initialMetrics: loadMetricsExplorerMetrics,
        timeRange
      }
    ),
    variableEditor ? /* @__PURE__ */ jsx(InlineFieldRow, { children: /* @__PURE__ */ jsx(
      InlineField,
      {
        label: t("grafana-prometheus.querybuilder.metric-combobox.label-metric", "Metric"),
        labelWidth: 20,
        tooltip: /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-prometheus.querybuilder.metric-combobox.tooltip-metric", children: "Optional: returns a list of label values for the label name in the specified metric." }) }),
        children: asyncSelect()
      }
    ) }) : /* @__PURE__ */ jsx(EditorFieldGroup, { children: /* @__PURE__ */ jsx(EditorField, { label: t("grafana-prometheus.querybuilder.metric-combobox.label-metric", "Metric"), children: asyncSelect() }) })
  ] });
}
const formatPrometheusLabelFiltersToString = (queryString, labelsFilters) => {
  const filterArray = labelsFilters ? formatPrometheusLabelFilters(labelsFilters) : [];
  return `{__name__=~".*${queryString}"${filterArray ? filterArray.join("") : ""}}`;
};
const formatPrometheusLabelFilters = (labelsFilters) => {
  return labelsFilters.map((label) => {
    return `,${label.label}="${label.value}"`;
  });
};
const formatKeyValueStrings = (query, labelsFilters) => {
  const queryString = regexifyLabelValuesQueryString(query);
  return formatPrometheusLabelFiltersToString(queryString, labelsFilters);
};

export { MetricCombobox, formatPrometheusLabelFilters, formatPrometheusLabelFiltersToString };
//# sourceMappingURL=MetricCombobox.mjs.map
