import { jsx } from 'react/jsx-runtime';
import * as H from 'history';
import React, { useContext } from 'react';
import { BehaviorSubject } from 'rxjs';
import { urlUtil, deprecationWarning } from '@grafana/data';
import { createLogger, attachDebugger } from '@grafana/ui';
import { config } from '../config.mjs';

"use strict";
class HistoryWrapper {
  constructor(history) {
    var _a;
    this.history = history || (process.env.NODE_ENV === "test" ? H.createMemoryHistory({ initialEntries: ["/"] }) : H.createBrowserHistory({ basename: (_a = config.appSubUrl) != null ? _a : "/" }));
    this.locationObservable = new BehaviorSubject(this.history.location);
    this.history.listen((location) => {
      this.locationObservable.next(location);
    });
    this.partial = this.partial.bind(this);
    this.push = this.push.bind(this);
    this.replace = this.replace.bind(this);
    this.getSearch = this.getSearch.bind(this);
    this.getHistory = this.getHistory.bind(this);
    this.getLocation = this.getLocation.bind(this);
  }
  getLocationObservable() {
    return this.locationObservable.asObservable();
  }
  getHistory() {
    return this.history;
  }
  getSearch() {
    return new URLSearchParams(this.history.location.search);
  }
  partial(query, replace) {
    const currentLocation = this.history.location;
    const newQuery = this.getSearchObject();
    for (const key in query) {
      if (query[key] === null || query[key] === void 0) {
        delete newQuery[key];
      } else {
        newQuery[key] = query[key];
      }
    }
    const updatedUrl = urlUtil.renderUrl(currentLocation.pathname, newQuery);
    if (replace) {
      this.history.replace(updatedUrl, this.history.location.state);
    } else {
      this.history.push(updatedUrl, this.history.location.state);
    }
  }
  push(location) {
    this.history.push(location);
  }
  replace(location) {
    this.history.replace(location);
  }
  reload() {
    var _a;
    const prevState = (_a = this.history.location.state) == null ? void 0 : _a.routeReloadCounter;
    this.history.replace({
      ...this.history.location,
      state: { routeReloadCounter: prevState ? prevState + 1 : 1 }
    });
  }
  getLocation() {
    return this.history.location;
  }
  getSearchObject() {
    return locationSearchToObject(this.history.location.search);
  }
  /** @deprecated use partial, push or replace instead */
  update(options) {
    deprecationWarning("LocationSrv", "update", "partial, push or replace");
    if (options.partial && options.query) {
      this.partial(options.query, options.partial);
    } else {
      const newLocation = {
        pathname: options.path
      };
      if (options.query) {
        newLocation.search = urlUtil.toUrlParams(options.query);
      }
      if (options.replace) {
        this.replace(newLocation);
      } else {
        this.push(newLocation);
      }
    }
  }
}
function locationSearchToObject(search) {
  let queryString = typeof search === "number" ? String(search) : search;
  if (queryString.length > 0) {
    if (queryString.startsWith("?")) {
      return urlUtil.parseKeyValue(queryString.substring(1));
    }
    return urlUtil.parseKeyValue(queryString);
  }
  return {};
}
let locationService = new HistoryWrapper();
const setLocationService = (location) => {
  if (process.env.NODE_ENV !== "test") {
    throw new Error("locationService can be only overriden in test environment");
  }
  locationService = location;
};
const navigationLog = createLogger("Router");
const navigationLogger = navigationLog.logger;
attachDebugger("location", locationService, navigationLog);
const LocationServiceContext = React.createContext(void 0);
function useLocationService() {
  const service = useContext(LocationServiceContext);
  if (!service) {
    throw new Error("useLocationService must be used within a LocationServiceProvider");
  }
  return service;
}
const LocationServiceProvider = ({
  service,
  children
}) => {
  return /* @__PURE__ */ jsx(LocationServiceContext.Provider, { value: service, children });
};

export { HistoryWrapper, LocationServiceProvider, locationSearchToObject, locationService, navigationLogger, setLocationService, useLocationService };
//# sourceMappingURL=LocationService.mjs.map
