import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import { map } from 'lodash';
import * as React from 'react';
import { CoreApp } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { t, Trans } from '@grafana/i18n';
import { EditorRow, EditorField, EditorSwitch } from '@grafana/plugin-ui';
import { AutoSizeInput, Select, RadioButtonGroup } from '@grafana/ui';
import { getQueryTypeOptions, getQueryTypeChangeHandler } from '../../components/PromExploreExtraField.mjs';
import { QueryOptionGroup } from '../shared/QueryOptionGroup.mjs';
import { PromQueryLegendEditor, getLegendModeLabel } from './PromQueryLegendEditor.mjs';

"use strict";
const INTERVAL_FACTOR_OPTIONS = map([1, 2, 3, 4, 5, 10], (value) => ({
  value,
  label: "1/" + value
}));
const PromQueryBuilderOptions = React.memo(
  ({ query, app, onChange, onRunQuery }) => {
    const FORMAT_OPTIONS = [
      {
        label: t(
          "grafana-prometheus.querybuilder.prom-query-builder-options.format-options.label-time-series",
          "Time series"
        ),
        value: "time_series"
      },
      {
        label: t("grafana-prometheus.querybuilder.prom-query-builder-options.format-options.label-table", "Table"),
        value: "table"
      },
      {
        label: t("grafana-prometheus.querybuilder.prom-query-builder-options.format-options.label-heatmap", "Heatmap"),
        value: "heatmap"
      }
    ];
    const onChangeFormat = (value) => {
      onChange({ ...query, format: value.value });
      onRunQuery();
    };
    const onChangeStep = (evt) => {
      onChange({ ...query, interval: evt.currentTarget.value.trim() });
      onRunQuery();
    };
    const queryTypeOptions = getQueryTypeOptions(
      app === CoreApp.Explore || app === CoreApp.Correlations || app === CoreApp.PanelEditor
    );
    const onQueryTypeChange = getQueryTypeChangeHandler(query, onChange);
    const onExemplarChange = (event) => {
      const isEnabled = event.currentTarget.checked;
      onChange({ ...query, exemplar: isEnabled });
      onRunQuery();
    };
    const onIntervalFactorChange = (value) => {
      onChange({ ...query, intervalFactor: value.value });
      onRunQuery();
    };
    const formatOption = FORMAT_OPTIONS.find((option) => option.value === query.format) || FORMAT_OPTIONS[0];
    const queryTypeValue = getQueryTypeValue(query);
    const queryTypeLabel = queryTypeOptions.find((x) => x.value === queryTypeValue).label;
    return /* @__PURE__ */ jsx(EditorRow, { children: /* @__PURE__ */ jsx("div", { "data-testid": selectors.components.DataSource.Prometheus.queryEditor.options, children: /* @__PURE__ */ jsxs(
      QueryOptionGroup,
      {
        title: t("grafana-prometheus.querybuilder.prom-query-builder-options.title-options", "Options"),
        collapsedInfo: getCollapsedInfo(query, formatOption.label, queryTypeLabel, app),
        children: [
          /* @__PURE__ */ jsx(
            PromQueryLegendEditor,
            {
              legendFormat: query.legendFormat,
              onChange: (legendFormat) => onChange({ ...query, legendFormat }),
              onRunQuery
            }
          ),
          /* @__PURE__ */ jsx(
            EditorField,
            {
              label: t("grafana-prometheus.querybuilder.prom-query-builder-options.label-min-step", "Min step"),
              tooltip: /* @__PURE__ */ jsx(Fragment, { children: /* @__PURE__ */ jsxs(
                Trans,
                {
                  i18nKey: "grafana-prometheus.querybuilder.prom-query-builder-options.tooltip-min-step",
                  values: {
                    interval: "$__interval",
                    rateInterval: "$__rate_interval"
                  },
                  children: [
                    "An additional lower limit for the step parameter of the Prometheus query and for the",
                    " ",
                    /* @__PURE__ */ jsx("code", { children: "{{interval}}" }),
                    " and ",
                    /* @__PURE__ */ jsx("code", { children: "{{rateInterval}}" }),
                    " variables."
                  ]
                }
              ) }),
              children: /* @__PURE__ */ jsx(
                AutoSizeInput,
                {
                  type: "text",
                  "aria-label": t(
                    "grafana-prometheus.querybuilder.prom-query-builder-options.aria-label-lower-limit-parameter",
                    "Set lower limit for the step parameter"
                  ),
                  placeholder: t("grafana-prometheus.querybuilder.prom-query-builder-options.placeholder-auto", "auto"),
                  minWidth: 10,
                  onCommitChange: onChangeStep,
                  defaultValue: query.interval,
                  "data-testid": selectors.components.DataSource.Prometheus.queryEditor.step
                }
              )
            }
          ),
          /* @__PURE__ */ jsx(EditorField, { label: t("grafana-prometheus.querybuilder.prom-query-builder-options.label-format", "Format"), children: /* @__PURE__ */ jsx(
            Select,
            {
              "data-testid": selectors.components.DataSource.Prometheus.queryEditor.format,
              value: formatOption,
              allowCustomValue: true,
              onChange: onChangeFormat,
              options: FORMAT_OPTIONS
            }
          ) }),
          /* @__PURE__ */ jsx(
            EditorField,
            {
              label: t("grafana-prometheus.querybuilder.prom-query-builder-options.label-type", "Type"),
              "data-testid": selectors.components.DataSource.Prometheus.queryEditor.type,
              children: /* @__PURE__ */ jsx(RadioButtonGroup, { options: queryTypeOptions, value: queryTypeValue, onChange: onQueryTypeChange })
            }
          ),
          shouldShowExemplarSwitch(query, app) && /* @__PURE__ */ jsx(
            EditorField,
            {
              label: t("grafana-prometheus.querybuilder.prom-query-builder-options.label-exemplars", "Exemplars"),
              children: /* @__PURE__ */ jsx(
                EditorSwitch,
                {
                  value: query.exemplar || false,
                  onChange: onExemplarChange,
                  "data-testid": selectors.components.DataSource.Prometheus.queryEditor.exemplars
                }
              )
            }
          ),
          query.intervalFactor && query.intervalFactor > 1 && /* @__PURE__ */ jsx(
            EditorField,
            {
              label: t("grafana-prometheus.querybuilder.prom-query-builder-options.label-resolution", "Resolution"),
              children: /* @__PURE__ */ jsx(
                Select,
                {
                  "aria-label": t(
                    "grafana-prometheus.querybuilder.prom-query-builder-options.aria-label-select-resolution",
                    "Select resolution"
                  ),
                  isSearchable: false,
                  options: INTERVAL_FACTOR_OPTIONS,
                  onChange: onIntervalFactorChange,
                  value: INTERVAL_FACTOR_OPTIONS.find((option) => option.value === query.intervalFactor)
                }
              )
            }
          )
        ]
      }
    ) }) });
  }
);
function shouldShowExemplarSwitch(query, app) {
  if (app === CoreApp.UnifiedAlerting || !query.range) {
    return false;
  }
  return true;
}
function getQueryTypeValue(query) {
  return query.range && query.instant ? "both" : query.instant ? "instant" : "range";
}
function getCollapsedInfo(query, formatOption, queryType, app) {
  var _a;
  const items = [];
  items.push(
    t("grafana-prometheus.querybuilder.get-collapsed-info.legend", "Legend: {{value}}", {
      value: getLegendModeLabel(query.legendFormat)
    })
  );
  items.push(
    t("grafana-prometheus.querybuilder.get-collapsed-info.format", "Format: {{value}}", { value: formatOption })
  );
  items.push(
    t("grafana-prometheus.querybuilder.get-collapsed-info.step", "Step: {{value}}", { value: (_a = query.interval) != null ? _a : "auto" })
  );
  items.push(t("grafana-prometheus.querybuilder.get-collapsed-info.type", "Type: {{value}}", { value: queryType }));
  if (shouldShowExemplarSwitch(query, app)) {
    items.push(
      t("grafana-prometheus.querybuilder.get-collapsed-info.exemplars", "Exemplars: {{value}}", {
        value: query.exemplar ? "true" : "false"
      })
    );
  }
  return items;
}
PromQueryBuilderOptions.displayName = "PromQueryBuilderOptions";

export { PromQueryBuilderOptions };
//# sourceMappingURL=PromQueryBuilderOptions.mjs.map
