import { jsx } from 'react/jsx-runtime';
import * as React from 'react';
import { useMemo, useRef, useEffect, useCallback } from 'react';
import { measureText } from '../../utils/measureText.mjs';
import { AutoSizeInputContext } from './AutoSizeInputContext.mjs';
import { Input } from './Input.mjs';

"use strict";
const AutoSizeInput = React.forwardRef((props, ref) => {
  const {
    defaultValue = "",
    minWidth = 10,
    maxWidth,
    onCommitChange,
    onChange,
    onKeyDown,
    onBlur,
    value: controlledValue,
    placeholder,
    ...restProps
  } = props;
  const [inputState, setInputValue] = useControlledState(controlledValue, onChange);
  const inputValue = inputState != null ? inputState : defaultValue;
  const inputWidth = useMemo(() => {
    const displayValue = inputValue || placeholder || "";
    const valueString = typeof displayValue === "string" ? displayValue : displayValue.toString();
    return getWidthFor(valueString, minWidth, maxWidth);
  }, [placeholder, inputValue, minWidth, maxWidth]);
  return /* @__PURE__ */ jsx(AutoSizeInputContext.Provider, { value: true, children: /* @__PURE__ */ jsx(
    Input,
    {
      "data-testid": "autosize-input",
      ...restProps,
      placeholder,
      ref,
      value: inputValue.toString(),
      onChange: (event) => {
        if (onChange) {
          onChange(event);
        }
        setInputValue(event.currentTarget.value);
      },
      width: inputWidth,
      onBlur: (event) => {
        if (onBlur) {
          onBlur(event);
        } else if (onCommitChange) {
          onCommitChange(event);
        }
      },
      onKeyDown: (event) => {
        if (onKeyDown) {
          onKeyDown(event);
        } else if (event.key === "Enter" && onCommitChange) {
          onCommitChange(event);
        }
      }
    }
  ) });
});
function getWidthFor(value, minWidth, maxWidth) {
  if (!value) {
    return minWidth;
  }
  const extraSpace = 3;
  const realWidth = measureText(value.toString(), 14).width / 8 + extraSpace;
  if (minWidth && realWidth < minWidth) {
    return minWidth;
  }
  if (maxWidth && realWidth > maxWidth) {
    return maxWidth;
  }
  return realWidth;
}
AutoSizeInput.displayName = "AutoSizeInput";
function useControlledState(controlledValue, onChange) {
  const isControlledNow = controlledValue !== void 0 && onChange !== void 0;
  const isControlledRef = useRef(isControlledNow);
  const hasLoggedControlledWarning = useRef(false);
  if (isControlledNow !== isControlledRef.current && !hasLoggedControlledWarning.current) {
    console.warn(
      "An AutoSizeInput is changing from an uncontrolled to a controlled input. If you want to control the input, the empty value should be an empty string."
    );
    hasLoggedControlledWarning.current = true;
  }
  const [internalValue, setInternalValue] = React.useState(controlledValue);
  useEffect(() => {
    if (!isControlledRef.current) {
      setInternalValue(controlledValue);
    }
  }, [controlledValue]);
  const handleChange = useCallback((newValue) => {
    if (!isControlledRef.current) {
      setInternalValue(newValue);
    }
  }, []);
  const value = isControlledRef.current ? controlledValue : internalValue;
  return [value, handleChange];
}

export { AutoSizeInput };
//# sourceMappingURL=AutoSizeInput.mjs.map
