import { getDefaultCacheHeaders } from './caching.mjs';
import { EMPTY_SELECTOR, MATCH_ALL_LABELS, METRIC_LABEL, DEFAULT_SERIES_LIMIT } from './constants.mjs';
import { removeQuotesIfExist } from './language_provider.mjs';
import { processHistogramMetrics, getRangeSnapInterval } from './language_utils.mjs';
import { buildVisualQueryFromString } from './querybuilder/parsing.mjs';
import { PrometheusCacheLevel } from './types.mjs';
import { escapeForUtf8Support, utf8Support } from './utf8_support.mjs';

"use strict";
class BaseResourceClient {
  constructor(request, datasource) {
    this.request = request;
    this.datasource = datasource;
    /**
     * Fetches all time series that match a specific label matcher using **series** endpoint.
     *
     * @param {TimeRange} timeRange - Time range to use for the query
     * @param {string} match - Label matcher to filter time series
     * @param {string} limit - Maximum number of series to return
     */
    this.querySeries = async (timeRange, match, limit) => {
      const effectiveMatch = !match || match === EMPTY_SELECTOR ? MATCH_ALL_LABELS : match;
      const timeParams = this.datasource.getTimeRangeParams(timeRange);
      const searchParams = { ...timeParams, "match[]": effectiveMatch, limit };
      return await this.requestSeries("/api/v1/series", searchParams, getDefaultCacheHeaders(this.datasource.cacheLevel));
    };
    this.seriesLimit = this.datasource.seriesLimit;
  }
  getEffectiveLimit(limit) {
    return limit || this.seriesLimit;
  }
  async requestLabels(url, params, options) {
    const response = await this.request(url, params, options);
    return Array.isArray(response) ? response : [];
  }
  async requestSeries(url, params, options) {
    const response = await this.request(url, params, options);
    return Array.isArray(response) ? response : [];
  }
}
class LabelsApiClient extends BaseResourceClient {
  constructor() {
    super(...arguments);
    this._cache = new ResourceClientsCache(this.datasource.cacheLevel);
    this.histogramMetrics = [];
    this.metrics = [];
    this.labelKeys = [];
    this.cachedLabelValues = {};
    this.start = async (timeRange) => {
      await this.queryMetrics(timeRange);
      this.labelKeys = await this.queryLabelKeys(timeRange);
    };
    this.queryMetrics = async (timeRange) => {
      this.metrics = await this.queryLabelValues(timeRange, METRIC_LABEL);
      this.histogramMetrics = processHistogramMetrics(this.metrics);
      this._cache.setLabelValues(timeRange, void 0, DEFAULT_SERIES_LIMIT, this.metrics);
      return { metrics: this.metrics, histogramMetrics: this.histogramMetrics };
    };
    /**
     * Fetches all available label keys from Prometheus using labels endpoint.
     * Uses the labels endpoint with optional match parameter for filtering.
     *
     * @param {TimeRange} timeRange - Time range to use for the query
     * @param {string} match - Optional label matcher to filter results
     * @param {string} limit - Maximum number of results to return
     * @returns {Promise<string[]>} Array of label keys sorted alphabetically
     */
    this.queryLabelKeys = async (timeRange, match, limit) => {
      let url = "/api/v1/labels";
      const timeParams = getRangeSnapInterval(this.datasource.cacheLevel, timeRange);
      const effectiveLimit = this.getEffectiveLimit(limit);
      const searchParams = { limit: effectiveLimit, ...timeParams, ...match ? { "match[]": match } : {} };
      const effectiveMatch = match != null ? match : "";
      const maybeCachedKeys = this._cache.getLabelKeys(timeRange, effectiveMatch, effectiveLimit);
      if (maybeCachedKeys) {
        return maybeCachedKeys;
      }
      const res = await this.requestLabels(url, searchParams, getDefaultCacheHeaders(this.datasource.cacheLevel));
      if (Array.isArray(res)) {
        this.labelKeys = res.slice().sort();
        this._cache.setLabelKeys(timeRange, effectiveMatch, effectiveLimit, this.labelKeys);
        return this.labelKeys.slice();
      }
      return [];
    };
    /**
     * Fetches all values for a specific label key from Prometheus using labels values endpoint.
     *
     * @param {TimeRange} timeRange - Time range to use for the query
     * @param {string} labelKey - The label key to fetch values for
     * @param {string} match - Optional label matcher to filter results
     * @param {string} limit - Maximum number of results to return
     * @returns {Promise<string[]>} Array of label values
     */
    this.queryLabelValues = async (timeRange, labelKey, match, limit) => {
      const timeParams = this.datasource.getAdjustedInterval(timeRange);
      const effectiveLimit = this.getEffectiveLimit(limit);
      const searchParams = { limit: effectiveLimit, ...timeParams, ...match ? { "match[]": match } : {} };
      const interpolatedName = this.datasource.interpolateString(labelKey);
      const interpolatedAndEscapedName = escapeForUtf8Support(removeQuotesIfExist(interpolatedName));
      const effectiveMatch = `${match != null ? match : ""}-${interpolatedAndEscapedName}`;
      const maybeCachedValues = this._cache.getLabelValues(timeRange, effectiveMatch, effectiveLimit);
      if (maybeCachedValues) {
        return maybeCachedValues;
      }
      const url = `/api/v1/label/${interpolatedAndEscapedName}/values`;
      const value = await this.requestLabels(url, searchParams, getDefaultCacheHeaders(this.datasource.cacheLevel));
      this._cache.setLabelValues(timeRange, effectiveMatch, effectiveLimit, value != null ? value : []);
      return value != null ? value : [];
    };
  }
}
class SeriesApiClient extends BaseResourceClient {
  constructor() {
    super(...arguments);
    this._cache = new ResourceClientsCache(this.datasource.cacheLevel);
    this.histogramMetrics = [];
    this.metrics = [];
    this.labelKeys = [];
    this.cachedLabelValues = {};
    this.start = async (timeRange) => {
      await this.queryMetrics(timeRange);
    };
    this.queryMetrics = async (timeRange) => {
      const series = await this.querySeries(timeRange, void 0, DEFAULT_SERIES_LIMIT);
      const { metrics, labelKeys } = processSeries(series, METRIC_LABEL);
      this.metrics = metrics;
      this.histogramMetrics = processHistogramMetrics(this.metrics);
      this.labelKeys = labelKeys;
      this._cache.setLabelValues(timeRange, void 0, DEFAULT_SERIES_LIMIT, metrics);
      this._cache.setLabelKeys(timeRange, void 0, DEFAULT_SERIES_LIMIT, labelKeys);
      return { metrics: this.metrics, histogramMetrics: this.histogramMetrics };
    };
    this.queryLabelKeys = async (timeRange, match, limit) => {
      const effectiveLimit = this.getEffectiveLimit(limit);
      const effectiveMatch = !match || match === EMPTY_SELECTOR ? void 0 : match;
      const maybeCachedKeys = this._cache.getLabelKeys(timeRange, effectiveMatch, effectiveLimit);
      if (maybeCachedKeys) {
        return maybeCachedKeys;
      }
      const series = await this.querySeries(timeRange, effectiveMatch, effectiveLimit);
      const { labelKeys } = processSeries(series);
      this._cache.setLabelKeys(timeRange, effectiveMatch, effectiveLimit, labelKeys);
      return labelKeys;
    };
    this.queryLabelValues = async (timeRange, labelKey, match, limit) => {
      let effectiveMatch = "";
      if (!match || match === EMPTY_SELECTOR) {
        effectiveMatch = `{${utf8Support(removeQuotesIfExist(labelKey))}!=""}`;
      } else {
        const {
          query: { metric, labels }
        } = buildVisualQueryFromString(match);
        labels.push({
          label: removeQuotesIfExist(labelKey),
          op: "!=",
          value: ""
        });
        const metricFilter = metric ? `__name__="${metric}",` : "";
        const labelFilters = labels.map((lf) => `${utf8Support(lf.label)}${lf.op}"${lf.value}"`).join(",");
        effectiveMatch = `{${metricFilter}${labelFilters}}`;
      }
      const effectiveLimit = this.getEffectiveLimit(limit);
      const maybeCachedValues = this._cache.getLabelValues(timeRange, effectiveMatch, effectiveLimit);
      if (maybeCachedValues) {
        return maybeCachedValues;
      }
      const series = await this.querySeries(timeRange, effectiveMatch, effectiveLimit);
      const { labelValues } = processSeries(series, removeQuotesIfExist(labelKey));
      this._cache.setLabelValues(timeRange, effectiveMatch, effectiveLimit, labelValues);
      return labelValues;
    };
  }
}
class ResourceClientsCache {
  constructor(cacheLevel = PrometheusCacheLevel.High) {
    this.cacheLevel = cacheLevel;
    this.MAX_CACHE_ENTRIES = 1e3;
    // Maximum number of cache entries
    this.MAX_CACHE_SIZE_BYTES = 50 * 1024 * 1024;
    // 50MB max cache size
    this._cache = {};
    this._accessTimestamps = {};
  }
  setLabelKeys(timeRange, match, limit, keys) {
    if (keys.length === 0) {
      return;
    }
    this.cleanCacheIfNeeded();
    const cacheKey = this.getCacheKey(timeRange, match, limit, "key");
    this._cache[cacheKey] = keys.slice().sort();
    this._accessTimestamps[cacheKey] = Date.now();
  }
  getLabelKeys(timeRange, match, limit) {
    const cacheKey = this.getCacheKey(timeRange, match, limit, "key");
    const result = this._cache[cacheKey];
    if (result) {
      this._accessTimestamps[cacheKey] = Date.now();
    }
    return result;
  }
  setLabelValues(timeRange, match, limit, values) {
    if (values.length === 0) {
      return;
    }
    this.cleanCacheIfNeeded();
    const cacheKey = this.getCacheKey(timeRange, match, limit, "value");
    this._cache[cacheKey] = values.slice().sort();
    this._accessTimestamps[cacheKey] = Date.now();
  }
  getLabelValues(timeRange, match, limit) {
    const cacheKey = this.getCacheKey(timeRange, match, limit, "value");
    const result = this._cache[cacheKey];
    if (result) {
      this._accessTimestamps[cacheKey] = Date.now();
    }
    return result;
  }
  getCacheKey(timeRange, match, limit, type) {
    const snappedTimeRange = getRangeSnapInterval(this.cacheLevel, timeRange);
    return [snappedTimeRange.start, snappedTimeRange.end, limit, match, type].join("|");
  }
  cleanCacheIfNeeded() {
    const currentEntries = Object.keys(this._cache).length;
    if (currentEntries >= this.MAX_CACHE_ENTRIES) {
      const entriesToRemove = Math.max(1, Math.floor(currentEntries - this.MAX_CACHE_ENTRIES + 1));
      this.removeOldestEntries(entriesToRemove);
    }
    const currentSize = this.getCacheSizeInBytes();
    if (currentSize > this.MAX_CACHE_SIZE_BYTES) {
      const entriesToRemove = Math.max(1, Math.floor(Object.keys(this._cache).length * 0.2));
      this.removeOldestEntries(entriesToRemove);
    }
  }
  getCacheSizeInBytes() {
    let size = 0;
    for (const key in this._cache) {
      size += key.length * 2;
      const value = this._cache[key];
      for (const item of value) {
        size += item.length * 2;
      }
    }
    return size;
  }
  removeOldestEntries(count) {
    const entries = Object.entries(this._accessTimestamps).sort(
      ([, timestamp1], [, timestamp2]) => timestamp1 - timestamp2
    );
    const entriesToRemove = entries.slice(0, count);
    for (const [key] of entriesToRemove) {
      delete this._cache[key];
      delete this._accessTimestamps[key];
    }
  }
}
function processSeries(series, findValuesForKey) {
  const metrics = /* @__PURE__ */ new Set();
  const labelKeys = /* @__PURE__ */ new Set();
  const labelValues = /* @__PURE__ */ new Set();
  series.forEach((item) => {
    if (METRIC_LABEL in item) {
      metrics.add(item.__name__);
    }
    Object.keys(item).forEach((key) => {
      if (key !== METRIC_LABEL) {
        labelKeys.add(key);
      }
      if (findValuesForKey && key === findValuesForKey) {
        labelValues.add(item[key]);
      }
    });
  });
  return {
    metrics: Array.from(metrics).sort(),
    labelKeys: Array.from(labelKeys).sort(),
    labelValues: Array.from(labelValues).sort()
  };
}

export { BaseResourceClient, LabelsApiClient, SeriesApiClient, processSeries };
//# sourceMappingURL=resource_clients.mjs.map
