import { jsx } from 'react/jsx-runtime';
import i18n from 'i18next';
import LanguageDetector from 'i18next-browser-languagedetector';
import { Trans as Trans$1, initReactI18next, getI18n, setDefaults, setI18n } from 'react-i18next';
import { PSEUDO_LOCALE, DEFAULT_LANGUAGE } from './constants.mjs';
import { initRegionalFormat } from './dates.mjs';
import { LANGUAGES } from './languages.mjs';

"use strict";
let tFunc;
let transComponent;
const VALID_LANGUAGES = [
  ...LANGUAGES,
  {
    name: "Pseudo",
    code: PSEUDO_LOCALE
  }
];
function initTFuncAndTransComponent({ id, ns } = {}) {
  if (id) {
    tFunc = getI18nInstance().getFixedT(null, id);
    transComponent = (props) => /* @__PURE__ */ jsx(Trans$1, { shouldUnescape: true, ns: id, ...props });
    return;
  }
  tFunc = getI18nInstance().t;
  transComponent = (props) => /* @__PURE__ */ jsx(Trans$1, { shouldUnescape: true, ns, ...props });
}
async function loadNamespacedResources(namespace, language, loaders) {
  if (!(loaders == null ? void 0 : loaders.length)) {
    return;
  }
  const resolvedLanguage = language === PSEUDO_LOCALE ? DEFAULT_LANGUAGE : language;
  return Promise.all(
    loaders.map(async (loader) => {
      try {
        const resources = await loader(resolvedLanguage);
        addResourceBundle(resolvedLanguage, namespace, resources);
      } catch (error) {
        console.error(`Error loading resources for namespace ${namespace} and language: ${resolvedLanguage}`, error);
      }
    })
  );
}
function initDefaultI18nInstance() {
  var _a;
  if (((_a = getI18nInstance().options) == null ? void 0 : _a.resources) && typeof getI18nInstance().options.resources === "object") {
    return;
  }
  const initPromise = getI18nInstance().use(initReactI18next).init({
    resources: {},
    returnEmptyString: false,
    lng: DEFAULT_LANGUAGE
    // this should be the locale of the phrases in our source JSX
  });
  initTFuncAndTransComponent();
  return initPromise;
}
function initDefaultReactI18nInstance() {
  var _a, _b;
  if ((_b = (_a = getI18n()) == null ? void 0 : _a.options) == null ? void 0 : _b.react) {
    return;
  }
  const options = {};
  setDefaults(options);
  setI18n(getI18nInstance());
}
async function initPluginTranslations(id, loaders) {
  await initDefaultI18nInstance();
  initDefaultReactI18nInstance();
  const language = getResolvedLanguage();
  initTFuncAndTransComponent({ id });
  await loadNamespacedResources(id, language, loaders);
  return { language };
}
function getI18nInstance() {
  return i18n;
}
async function initTranslations({
  ns,
  language = DEFAULT_LANGUAGE,
  module
}) {
  var _a, _b;
  const options = {
    // We don't bundle any translations, we load them async
    partialBundledLanguages: true,
    resources: {},
    // If translations are empty strings (no translation), fall back to the default value in source code
    returnEmptyString: false,
    // Required to ensure that `resolvedLanguage` is set property when an invalid language is passed (such as through 'detect')
    supportedLngs: VALID_LANGUAGES.map((lang) => lang.code),
    fallbackLng: DEFAULT_LANGUAGE,
    ns,
    postProcess: [
      // Add pseudo processing even if we aren't necessarily going to use it
      PSEUDO_LOCALE
    ]
  };
  if (language === "detect") {
    getI18nInstance().use(LanguageDetector);
    const detection = { order: ["navigator"], caches: [] };
    options.detection = detection;
  } else {
    options.lng = (_b = (_a = VALID_LANGUAGES.find((lang) => lang.code === language)) == null ? void 0 : _a.code) != null ? _b : void 0;
  }
  if (module) {
    getI18nInstance().use(module).use(initReactI18next);
  } else {
    getI18nInstance().use(initReactI18next);
  }
  if (language === PSEUDO_LOCALE) {
    const { default: Pseudo } = await import('i18next-pseudo');
    getI18nInstance().use(
      new Pseudo({
        languageToPseudo: PSEUDO_LOCALE,
        enabled: true,
        wrapped: true
      })
    );
  }
  await getI18nInstance().init(options);
  initTFuncAndTransComponent({ ns });
  return {
    language: getResolvedLanguage()
  };
}
function getLanguage() {
  var _a;
  return ((_a = getI18nInstance()) == null ? void 0 : _a.language) || DEFAULT_LANGUAGE;
}
function getResolvedLanguage() {
  var _a;
  return ((_a = getI18nInstance()) == null ? void 0 : _a.resolvedLanguage) || DEFAULT_LANGUAGE;
}
function getNamespaces() {
  var _a;
  return (_a = getI18nInstance()) == null ? void 0 : _a.options.ns;
}
async function changeLanguage(language) {
  var _a, _b;
  const validLanguage = (_b = (_a = VALID_LANGUAGES.find((lang) => lang.code === language)) == null ? void 0 : _a.code) != null ? _b : DEFAULT_LANGUAGE;
  await getI18nInstance().changeLanguage(validLanguage);
}
async function initializeI18n({ language, ns, module }, regionalFormat) {
  initRegionalFormat(regionalFormat);
  return initTranslations({ language, ns, module });
}
function addResourceBundle(language, namespace, resources) {
  getI18nInstance().addResourceBundle(language, namespace, resources, true, false);
}
const t = (id, defaultMessage, values) => {
  initDefaultI18nInstance();
  if (!tFunc) {
    if (process.env.NODE_ENV !== "test") {
      console.warn(
        "t() was called before i18n was initialized. This is probably caused by calling t() in the root module scope, instead of lazily on render"
      );
    }
    if (process.env.NODE_ENV === "development") {
      throw new Error("t() was called before i18n was initialized");
    }
    tFunc = getI18nInstance().t;
  }
  return tFunc(id, defaultMessage, values);
};
function Trans(props) {
  initDefaultI18nInstance();
  const Component = transComponent != null ? transComponent : Trans$1;
  return /* @__PURE__ */ jsx(Component, { shouldUnescape: true, ...props });
}

export { Trans, addResourceBundle, changeLanguage, getI18nInstance, getLanguage, getNamespaces, getResolvedLanguage, initDefaultI18nInstance, initDefaultReactI18nInstance, initPluginTranslations, initializeI18n, loadNamespacedResources, t };
//# sourceMappingURL=i18n.mjs.map
