import { jsx } from 'react/jsx-runtime';
import { useMemo, useState, useCallback } from 'react';
import { toOption, getFrameDisplayName, FieldMatcherID, fieldMatchers } from '@grafana/data';
import { Select, MultiSelect } from '../Select/Select.mjs';

"use strict";
const recoverRefIdMissing = (newRefIds, oldRefIds, previousValue) => {
  if (!previousValue) {
    return;
  }
  let changedTo = newRefIds.find((refId) => {
    return !oldRefIds.some((refId2) => {
      return refId === refId2;
    });
  });
  if (changedTo) {
    return changedTo;
  }
  return;
};
function RefIDPicker({ value, data, onChange, placeholder }) {
  const listOfRefIds = useMemo(() => getListOfQueryRefIds(data), [data]);
  const [priorSelectionState, updatePriorSelectionState] = useState({
    refIds: [],
    value: void 0
  });
  const currentValue = useMemo(() => {
    var _a;
    return (_a = listOfRefIds.find((refId) => refId.value === value)) != null ? _a : recoverRefIdMissing(listOfRefIds, priorSelectionState.refIds, priorSelectionState.value);
  }, [value, listOfRefIds, priorSelectionState]);
  const onFilterChange = useCallback(
    (v) => {
      onChange(v == null ? void 0 : v.value);
    },
    [onChange]
  );
  if (listOfRefIds !== priorSelectionState.refIds || (currentValue == null ? void 0 : currentValue.value) !== priorSelectionState.value) {
    updatePriorSelectionState({
      refIds: listOfRefIds,
      value: currentValue == null ? void 0 : currentValue.value
    });
  }
  return /* @__PURE__ */ jsx(
    Select,
    {
      options: listOfRefIds,
      onChange: onFilterChange,
      isClearable: true,
      placeholder: placeholder != null ? placeholder : "Select query refId",
      value: currentValue
    }
  );
}
const recoverMultiRefIdMissing = (newRefIds, oldRefIds, previousValue) => {
  if (!previousValue || !previousValue.length) {
    return;
  }
  const changedTo = newRefIds.filter((newRefId) => {
    return oldRefIds.some((oldRefId) => {
      return newRefId === oldRefId;
    });
  });
  if (changedTo.length) {
    return changedTo;
  }
  return;
};
function RefIDMultiPicker({ value, data, onChange, placeholder }) {
  var _a;
  const listOfRefIds = useMemo(() => getListOfQueryRefIds(data), [data]);
  const [priorSelectionState, updatePriorSelectionState] = useState({
    refIds: [],
    value: void 0
  });
  const currentValue = useMemo(() => {
    let extractedRefIds = /* @__PURE__ */ new Set();
    if (value) {
      if (value.startsWith("/^")) {
        try {
          extractedRefIds = new Set(regexpToStrings(value));
        } catch (e) {
          extractedRefIds.add(value);
        }
      } else if (value.includes("|")) {
        extractedRefIds = new Set(value.split("|"));
      } else {
        extractedRefIds.add(value);
      }
    }
    const matchedRefIds = listOfRefIds.filter((refId) => extractedRefIds.has(refId.value || ""));
    if (matchedRefIds.length) {
      return matchedRefIds;
    }
    const newRefIds = [...extractedRefIds].map(toOption);
    return recoverMultiRefIdMissing(newRefIds, priorSelectionState.refIds, priorSelectionState.value);
  }, [value, listOfRefIds, priorSelectionState]);
  const onFilterChange = useCallback(
    (v) => {
      onChange(v.map((v2) => v2.value));
    },
    [onChange]
  );
  if (listOfRefIds !== priorSelectionState.refIds || (currentValue == null ? void 0 : currentValue.length) !== ((_a = priorSelectionState.value) == null ? void 0 : _a.length)) {
    updatePriorSelectionState({
      refIds: listOfRefIds,
      value: currentValue
    });
  }
  return /* @__PURE__ */ jsx(
    MultiSelect,
    {
      options: listOfRefIds,
      onChange: onFilterChange,
      isClearable: true,
      placeholder: placeholder != null ? placeholder : "Select query refId",
      value: currentValue
    }
  );
}
function getListOfQueryRefIds(data) {
  var _a, _b;
  const queries = /* @__PURE__ */ new Map();
  for (const frame of data) {
    const refId = (_a = frame.refId) != null ? _a : "";
    const frames = (_b = queries.get(refId)) != null ? _b : [];
    if (frames.length === 0) {
      queries.set(refId, frames);
    }
    frames.push(frame);
  }
  const values = [];
  for (const [refId, frames] of queries.entries()) {
    values.push({
      value: refId,
      label: `Query: ${refId != null ? refId : "(missing refId)"}`,
      description: getFramesDescription(frames)
    });
  }
  return values;
}
function getFramesDescription(frames) {
  return `Frames (${frames.length}):
    ${frames.slice(0, Math.min(3, frames.length)).map((x) => getFrameDisplayName(x)).join(", ")} ${frames.length > 3 ? "..." : ""}`;
}
const fieldsByFrameRefIdItem = {
  id: FieldMatcherID.byFrameRefID,
  component: (props) => {
    return /* @__PURE__ */ jsx(RefIDPicker, { value: props.options, data: props.data, onChange: props.onChange });
  },
  matcher: fieldMatchers.get(FieldMatcherID.byFrameRefID),
  name: "Fields returned by query",
  description: "Set properties for fields from a specific query",
  optionsToLabel: (options) => options
};
function escapeRegExp(string) {
  return string.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
}
const regexpToStrings = (regexp) => {
  return regexp.slice(5, -3).split(/(?<!\\)\|/g).map((string) => string.replace(/\\(.)/g, "$1"));
};
const stringsToRegexp = (strings) => {
  return `/^(?:${strings.map((string) => escapeRegExp(string)).join("|")})$/`;
};

export { RefIDMultiPicker, RefIDPicker, fieldsByFrameRefIdItem, regexpToStrings, stringsToRegexp };
//# sourceMappingURL=FieldsByFrameRefIdMatcher.mjs.map
