import { jsxs, jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import { isEqual } from 'lodash';
import { memo, useCallback } from 'react';
import { usePrevious } from 'react-use';
import { useStyles2, InlineFormLabel, RadioButtonGroup } from '@grafana/ui';
import { PromExemplarField } from './PromExemplarField.mjs';

"use strict";
const PromExploreExtraField = memo(({ query, datasource, onChange, onRunQuery }) => {
  var _a;
  const rangeOptions = getQueryTypeOptions(true);
  const prevQuery = usePrevious(query);
  const styles = useStyles2(getStyles);
  const onExemplarChange = useCallback(
    (exemplar) => {
      if (!isEqual(query, prevQuery) || exemplar !== query.exemplar) {
        onChange({ ...query, exemplar });
      }
    },
    [prevQuery, query, onChange]
  );
  function onChangeQueryStep(interval) {
    onChange({ ...query, interval });
  }
  function onStepChange(e) {
    if (e.currentTarget.value !== query.interval) {
      onChangeQueryStep(e.currentTarget.value);
    }
  }
  function onReturnKeyDown(e) {
    if (e.key === "Enter" && e.shiftKey) {
      onRunQuery();
    }
  }
  const onQueryTypeChange = getQueryTypeChangeHandler(query, onChange);
  return /* @__PURE__ */ jsxs(
    "div",
    {
      "aria-label": "Prometheus extra field",
      className: "gf-form-inline",
      "data-testid": promExploreExtraFieldTestIds.extraFieldEditor,
      children: [
        /* @__PURE__ */ jsxs(
          "div",
          {
            "data-testid": promExploreExtraFieldTestIds.queryTypeField,
            className: cx(
              "gf-form",
              styles.queryTypeField,
              css({
                flexWrap: "nowrap"
              })
            ),
            "aria-label": "Query type field",
            children: [
              /* @__PURE__ */ jsx(InlineFormLabel, { width: "auto", children: "Query type" }),
              /* @__PURE__ */ jsx(
                RadioButtonGroup,
                {
                  options: rangeOptions,
                  value: query.range && query.instant ? "both" : query.instant ? "instant" : "range",
                  onChange: onQueryTypeChange
                }
              )
            ]
          }
        ),
        /* @__PURE__ */ jsxs(
          "div",
          {
            "data-testid": promExploreExtraFieldTestIds.stepField,
            className: cx(
              "gf-form",
              css({
                flexWrap: "nowrap"
              })
            ),
            "aria-label": "Step field",
            children: [
              /* @__PURE__ */ jsx(
                InlineFormLabel,
                {
                  width: 6,
                  tooltip: "Time units and built-in variables can be used here, for example: $__interval, $__rate_interval, 5s, 1m, 3h, 1d, 1y (Default if no unit is specified: s)",
                  children: "Min step"
                }
              ),
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: "text",
                  className: "gf-form-input width-4",
                  placeholder: "auto",
                  onChange: onStepChange,
                  onKeyDown: onReturnKeyDown,
                  value: (_a = query.interval) != null ? _a : ""
                }
              )
            ]
          }
        ),
        /* @__PURE__ */ jsx(PromExemplarField, { onChange: onExemplarChange, datasource, query })
      ]
    }
  );
});
PromExploreExtraField.displayName = "PromExploreExtraField";
function getQueryTypeOptions(includeBoth) {
  const rangeOptions = [
    { value: "range", label: "Range", description: "Run query over a range of time" },
    {
      value: "instant",
      label: "Instant",
      description: 'Run query against a single point in time. For this query, the "To" time is used'
    }
  ];
  if (includeBoth) {
    rangeOptions.push({ value: "both", label: "Both", description: "Run an Instant query and a Range query" });
  }
  return rangeOptions;
}
function getQueryTypeChangeHandler(query, onChange) {
  return (queryType) => {
    if (queryType === "instant") {
      onChange({ ...query, instant: true, range: false, exemplar: false });
    } else if (queryType === "range") {
      onChange({ ...query, instant: false, range: true });
    } else {
      onChange({ ...query, instant: true, range: true });
    }
  };
}
const promExploreExtraFieldTestIds = {
  extraFieldEditor: "prom-editor-extra-field",
  stepField: "prom-editor-extra-field-step",
  queryTypeField: "prom-editor-extra-field-query-type"
};
const getStyles = (theme) => ({
  queryTypeField: css({
    marginRight: theme.spacing(0.5)
  })
});

export { PromExploreExtraField, getQueryTypeChangeHandler, getQueryTypeOptions, promExploreExtraFieldTestIds };
//# sourceMappingURL=PromExploreExtraField.mjs.map
