import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { map, isEqual } from 'lodash';
import { memo, useState, useCallback, useEffect } from 'react';
import { CoreApp, LoadingState } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { EditorHeader, FlexItem, EditorRows } from '@grafana/plugin-ui';
import { reportInteraction } from '@grafana/runtime';
import { ConfirmModal, Button, Space } from '@grafana/ui';
import { QueryPatternsModal } from '../QueryPatternsModal.mjs';
import { useFlag, promQueryEditorExplainKey } from '../hooks/useFlag.mjs';
import { buildVisualQueryFromString } from '../parsing.mjs';
import { QueryEditorModeToggle } from '../shared/QueryEditorModeToggle.mjs';
import { QueryHeaderSwitch } from '../shared/QueryHeaderSwitch.mjs';
import { QueryEditorMode } from '../shared/types.mjs';
import { getQueryWithDefaults, changeEditorMode } from '../state.mjs';
import { PromQueryBuilderContainer } from './PromQueryBuilderContainer.mjs';
import { PromQueryBuilderOptions } from './PromQueryBuilderOptions.mjs';
import { PromQueryCodeEditor } from './PromQueryCodeEditor.mjs';
import { PromQueryCodeEditorAutocompleteInfo } from './PromQueryCodeEditorAutocompleteInfo.mjs';

"use strict";
const FORMAT_OPTIONS = [
  { label: "Time series", value: "time_series" },
  { label: "Table", value: "table" },
  { label: "Heatmap", value: "heatmap" }
];
const INTERVAL_FACTOR_OPTIONS = map([1, 2, 3, 4, 5, 10], (value) => ({
  value,
  label: "1/" + value
}));
const PromQueryEditorSelector = memo((props) => {
  const {
    onChange,
    onRunQuery,
    data,
    app,
    onAddQuery,
    datasource: { defaultEditor },
    queries
  } = props;
  const [parseModalOpen, setParseModalOpen] = useState(false);
  const [queryPatternsModalOpen, setQueryPatternsModalOpen] = useState(false);
  const [dataIsStale, setDataIsStale] = useState(false);
  const { flag: explain, setFlag: setExplain } = useFlag(promQueryEditorExplainKey);
  const query = getQueryWithDefaults(props.query, app, defaultEditor);
  const editorMode = query.editorMode;
  const onEditorModeChange = useCallback(
    (newMetricEditorMode) => {
      var _a;
      reportInteraction("user_grafana_prometheus_editor_mode_clicked", {
        newEditor: newMetricEditorMode,
        previousEditor: (_a = query.editorMode) != null ? _a : "",
        newQuery: !query.expr,
        app: app != null ? app : ""
      });
      if (newMetricEditorMode === QueryEditorMode.Builder) {
        const result = buildVisualQueryFromString(query.expr || "");
        if (result.errors.length) {
          setParseModalOpen(true);
          return;
        }
      }
      changeEditorMode(query, newMetricEditorMode, onChange);
    },
    [onChange, query, app]
  );
  useEffect(() => {
    setDataIsStale(false);
  }, [data]);
  const onChangeInternal = (query2) => {
    if (!isEqual(query2, props.query)) {
      setDataIsStale(true);
    }
    onChange(query2);
  };
  const onShowExplainChange = (e) => {
    setExplain(e.currentTarget.checked);
  };
  const handleOpenQueryPatternsModal = useCallback(() => {
    reportInteraction("grafana_prometheus_open_kickstart_clicked", {
      app: app != null ? app : ""
    });
    setQueryPatternsModalOpen(true);
  }, [app]);
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsx(
      ConfirmModal,
      {
        isOpen: parseModalOpen,
        title: "Parsing error: Switch to the builder mode?",
        body: "There is a syntax error, or the query structure cannot be visualized when switching to the builder mode. Parts of the query may be lost. ",
        confirmText: "Continue",
        onConfirm: () => {
          changeEditorMode(query, QueryEditorMode.Builder, onChange);
          setParseModalOpen(false);
        },
        onDismiss: () => setParseModalOpen(false)
      }
    ),
    /* @__PURE__ */ jsx(
      QueryPatternsModal,
      {
        isOpen: queryPatternsModalOpen,
        onClose: () => setQueryPatternsModalOpen(false),
        query,
        queries,
        app,
        onChange,
        onAddQuery
      }
    ),
    /* @__PURE__ */ jsxs(EditorHeader, { children: [
      /* @__PURE__ */ jsx(
        Button,
        {
          "data-testid": selectors.components.QueryBuilder.queryPatterns,
          variant: "secondary",
          size: "sm",
          onClick: handleOpenQueryPatternsModal,
          children: "Kick start your query"
        }
      ),
      /* @__PURE__ */ jsx("div", { "data-testid": selectors.components.DataSource.Prometheus.queryEditor.explain, children: /* @__PURE__ */ jsx(QueryHeaderSwitch, { label: "Explain", value: explain, onChange: onShowExplainChange }) }),
      /* @__PURE__ */ jsx(FlexItem, { grow: 1 }),
      app !== CoreApp.Explore && app !== CoreApp.Correlations && /* @__PURE__ */ jsx(
        Button,
        {
          variant: dataIsStale ? "primary" : "secondary",
          size: "sm",
          onClick: onRunQuery,
          icon: (data == null ? void 0 : data.state) === LoadingState.Loading ? "spinner" : void 0,
          disabled: (data == null ? void 0 : data.state) === LoadingState.Loading,
          children: "Run queries"
        }
      ),
      /* @__PURE__ */ jsx(PromQueryCodeEditorAutocompleteInfo, { datasourceUid: props.datasource.uid, editorMode }),
      /* @__PURE__ */ jsx("div", { "data-testid": selectors.components.DataSource.Prometheus.queryEditor.editorToggle, children: /* @__PURE__ */ jsx(QueryEditorModeToggle, { mode: editorMode, onChange: onEditorModeChange }) })
    ] }),
    /* @__PURE__ */ jsx(Space, { v: 0.5 }),
    /* @__PURE__ */ jsxs(EditorRows, { children: [
      editorMode === QueryEditorMode.Code && /* @__PURE__ */ jsx(PromQueryCodeEditor, { ...props, query, showExplain: explain, onChange: onChangeInternal }),
      editorMode === QueryEditorMode.Builder && /* @__PURE__ */ jsx(
        PromQueryBuilderContainer,
        {
          query,
          datasource: props.datasource,
          onChange: onChangeInternal,
          onRunQuery: props.onRunQuery,
          data,
          showExplain: explain
        }
      ),
      /* @__PURE__ */ jsx(PromQueryBuilderOptions, { query, app: props.app, onChange, onRunQuery })
    ] })
  ] });
});
PromQueryEditorSelector.displayName = "PromQueryEditorSelector";

export { FORMAT_OPTIONS, INTERVAL_FACTOR_OPTIONS, PromQueryEditorSelector };
//# sourceMappingURL=PromQueryEditorSelector.mjs.map
