import { screen, waitFor } from '@testing-library/react';
import { useForm, FormProvider } from 'react-hook-form';
import { render } from 'test/test-utils';

import {
  SceneTimeRange,
  SceneVariableSet,
  TestVariable,
  VariableValueSelectors,
  SceneDataLayerControls,
} from '@grafana/scenes';
import { ReportBaseV2 } from 'app/extensions/types';
import { DashboardScene } from 'app/features/dashboard-scene/scene/DashboardScene';

import { SelectDashboardScene } from './SelectDashboardScene';
import SelectDashboards from './SelectDashboards';

jest.mock('app/core/core', () => {
  return {
    ...jest.requireActual('app/core/core'),
    contextSrv: {
      hasPermission: () => true,
    },
  };
});

jest.mock('app/features/dashboard/api/dashboard_api', () => ({
  getDashboardAPI: () => ({
    getDashboardDTO: jest.fn().mockResolvedValue({
      dashboard: {
        uid: 'test-dashboard',
        title: 'Test Dashboard',
      },
      meta: {
        folderTitle: 'Test Folder',
        folderUid: 'test-folder',
      },
    }),
  }),
}));

jest.mock('app/core/services/backend_srv', () => ({
  backendSrv: {
    search: jest.fn().mockResolvedValue([]),
  },
}));

const mockOnAddDashboard = jest.fn();
const mockOnRemoveDashboard = jest.fn();

const getSelectDashboardScene = () => {
  const varA = new TestVariable({ name: 'A', query: 'A.*', value: 'A.AA', text: '', options: [], delayMs: 0 });
  const testDashboard = new DashboardScene({
    uid: 'test-dashboard',
    title: 'Test Dashboard',
    $timeRange: new SceneTimeRange({
      timeZone: 'browser',
    }),
    $variables: new SceneVariableSet({ variables: [varA] }),
  });

  return new SelectDashboardScene({
    uid: testDashboard.state.uid,
    title: testDashboard.state.title,
    $timeRange: testDashboard.state.$timeRange?.clone(),
    $variables: testDashboard.state.$variables?.clone(),
    variableControls: [new VariableValueSelectors({ layout: 'vertical' }), new SceneDataLayerControls()],
  });
};

const SelectDashboardsWrapper = ({ dashboards }: { dashboards?: SelectDashboardScene[] }) => {
  const methods = useForm<ReportBaseV2>({
    defaultValues: {
      dashboards: dashboards?.map((d) => ({ uid: d.state.uid, key: d.state.key })) || [],
    },
  });

  return (
    <FormProvider {...methods}>
      <SelectDashboards dashboards={dashboards || []} onAddDashboard={mockOnAddDashboard} />
    </FormProvider>
  );
};

async function setup(dashboards: SelectDashboardScene[] = []) {
  const { user, ...rest } = render(<SelectDashboardsWrapper dashboards={dashboards} />);

  await user.click(screen.getByRole('button', { name: /dashboard/i }));

  return {
    ...rest,
    user,
  };
}

describe('SelectDashboards', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  it('should render', async () => {
    await setup([getSelectDashboardScene()]);

    await waitFor(() => {
      expect(screen.getByRole('button', { name: /add dashboard/i })).toBeInTheDocument();
      expect(screen.getAllByText(/source dashboard/i)).toHaveLength(1);
    });
  });

  it('should call onAddDashboard when clicking add button', async () => {
    const { user } = await setup();

    await user.click(screen.getByRole('button', { name: /add dashboard/i }));

    expect(mockOnAddDashboard).toHaveBeenCalled();
  });

  it('should call onRemoveDashboard with correct index when removing dashboard', async () => {
    const varA = new TestVariable({
      name: 'A',
      query: 'A.*',
      value: 'A.AA',
      text: '',
      options: [],
      delayMs: 0,
    });

    const dashboards = [
      new SelectDashboardScene({
        uid: 'test-dashboard-1',
        title: 'Test Dashboard 1',
        $variables: new SceneVariableSet({ variables: [varA.clone()] }),
        variableControls: [new VariableValueSelectors({ layout: 'vertical' }), new SceneDataLayerControls()],
        onRemoveClick: mockOnRemoveDashboard,
      }),
      new SelectDashboardScene({
        uid: 'test-dashboard-2',
        title: 'Test Dashboard 2',
        $variables: new SceneVariableSet({ variables: [varA.clone()] }),
        variableControls: [new VariableValueSelectors({ layout: 'vertical' }), new SceneDataLayerControls()],
      }),
    ];

    const { user } = await setup(dashboards);

    await user.click(screen.getAllByRole('button', { name: /delete this dashboard/i })[0]);

    expect(mockOnRemoveDashboard).toHaveBeenCalledWith(dashboards[0]);
  });

  it('should show template variables section when dashboard has variables', async () => {
    const varA = new TestVariable({ name: 'A', query: 'A.*', value: 'A.AA', text: '', options: [], delayMs: 0 });

    await setup([
      new SelectDashboardScene({
        uid: 'test-dashboard',
        title: 'Test Dashboard',
        $timeRange: new SceneTimeRange({
          timeZone: 'browser',
        }),
        $variables: new SceneVariableSet({ variables: [varA] }),
        variableControls: [new VariableValueSelectors({ layout: 'vertical' }), new SceneDataLayerControls()],
      }),
    ]);

    expect(screen.getByText(/customize template variables/i)).toBeInTheDocument();
  });

  it('should show temp variables warning when same dashboard is added multiple times', async () => {
    const dashboards = Array(3)
      .fill(null)
      .map(() => {
        const varA = new TestVariable({
          name: 'A',
          query: 'A.*',
          value: 'A.AA',
          text: '',
          options: [],
          delayMs: 0,
        });

        return new SelectDashboardScene({
          uid: 'test-dashboard', // Same UID to simulate same dashboard
          title: 'Test Dashboard',
          $timeRange: new SceneTimeRange({
            timeZone: 'browser',
          }),
          $variables: new SceneVariableSet({ variables: [varA] }),
          variableControls: [new VariableValueSelectors({ layout: 'vertical' }), new SceneDataLayerControls()],
        });
      });

    await setup(dashboards);

    const alerts = screen.getAllByText(/template variables that you selected first are applied to all instances/i);
    expect(alerts).toHaveLength(2);
  });
});
