/**
 * Hook type for translation function that takes an ID, default message, and optional values
 * @returns A function that returns the translated string
 */
type UseTranslateHook = () => (id: string, defaultMessage: string, values?: Record<string, unknown>) => string;
/**
 * Type for children elements in Trans component
 * Can be either React nodes or an object of values
 */
type TransChild = React.ReactNode | Record<string, unknown>;
/**
 * Props interface for the Trans component used for internationalization
 */
interface TransProps {
    /**
     * The translation key to look up
     */
    i18nKey: string;
    /**
     * Child elements or values to interpolate
     */
    children?: TransChild | readonly TransChild[];
    /**
     * React elements to use for interpolation
     */
    components?: readonly React.ReactElement[] | {
        readonly [tagName: string]: React.ReactElement;
    };
    /**
     * Count value for pluralization
     */
    count?: number;
    /**
     * Default text if translation is not found
     */
    defaults?: string;
    /**
     * Namespace for the translation key
     */
    ns?: string;
    /**
     * Whether to unescape HTML entities
     */
    shouldUnescape?: boolean;
    /**
     * Values to interpolate into the translation
     */
    values?: Record<string, unknown>;
}
/**
 * Function declaration for the Trans component
 * @param props - The TransProps object containing translation configuration
 * @returns A React element with translated content
 */
declare function Trans$1(props: TransProps): React.ReactElement;
/**
 * Type alias for the Trans component
 */
type TransType = typeof Trans$1;

/**
 * Provides a i18next-compatible translation function.
 */
declare let useTranslate: UseTranslateHook;
declare function setUseTranslateHook(hook: UseTranslateHook): void;
/**
 * Sets the Trans component that will be used for translations throughout the application.
 * This function should only be called once during application initialization.
 *
 * @param transComponent - The Trans component function to use for translations
 * @throws {Error} If called multiple times outside of test environment
 */
declare function setTransComponent(transComponent: TransType): void;
/**
 * A React component for handling translations with support for interpolation and pluralization.
 * This component must be initialized using setTransComponent before use.
 *
 * @param props - The translation props including the i18nKey and any interpolation values
 * @returns A React element containing the translated content
 * @throws {Error} If the Trans component hasn't been initialized
 */
declare function Trans(props: TransProps): React.ReactElement;

export { Trans, type TransProps, setTransComponent, setUseTranslateHook, useTranslate };
