import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { useMemo, useState, useCallback, useId } from 'react';
import { selectors } from '@grafana/e2e-selectors';
import '../../themes/index.mjs';
import { t, Trans } from '../../utils/i18n.mjs';
import { Alert } from '../Alert/Alert.mjs';
import '../Button/index.mjs';
import { Field } from '../Forms/Field.mjs';
import { InlineField } from '../Forms/InlineField.mjs';
import { RadioButtonGroup } from '../Forms/RadioButtonGroup/RadioButtonGroup.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { Input } from '../Input/Input.mjs';
import { Stack } from '../Layout/Stack/Stack.mjs';
import { InlineSwitch } from '../Switch/Switch.mjs';
import { TagsInput } from '../TagsInput/TagsInput.mjs';
import { Text } from '../Text/Text.mjs';
import { BasicAuthSettings } from './BasicAuthSettings.mjs';
import { CustomHeadersSettings } from './CustomHeadersSettings.mjs';
import { HttpProxySettings } from './HttpProxySettings.mjs';
import { SecureSocksProxySettings } from './SecureSocksProxySettings.mjs';
import { TLSAuthSettings } from './TLSAuthSettings.mjs';
import { useTheme2 } from '../../themes/ThemeContext.mjs';
import { Button } from '../Button/Button.mjs';

"use strict";
const ACCESS_HELP_ID = "grafana-http-access-help";
const HttpAccessHelp = () => {
  return /* @__PURE__ */ jsxs(
    Alert,
    {
      severity: "info",
      title: t("grafana-ui.data-source-http-settings.access-help-title", "Access help"),
      topSpacing: 3,
      id: ACCESS_HELP_ID,
      children: [
        /* @__PURE__ */ jsx("p", { children: /* @__PURE__ */ jsxs(Trans, { i18nKey: "grafana-ui.data-source-http-settings.access-help-details", children: [
          "Access mode controls how requests to the data source will be handled.",
          /* @__PURE__ */ jsxs("strong", { children: [
            "\xA0",
            /* @__PURE__ */ jsx("i", { children: "Server" })
          ] }),
          " ",
          "should be the preferred way if nothing else is stated."
        ] }) }),
        /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.server-mode-title", children: /* @__PURE__ */ jsx(Text, { weight: "medium", children: "Server access mode (Default):" }) }),
        /* @__PURE__ */ jsx("p", { children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.server-mode-description", children: "All requests will be made from the browser to Grafana backend/server which in turn will forward the requests to the data source and by that circumvent possible Cross-Origin Resource Sharing (CORS) requirements. The URL needs to be accessible from the grafana backend/server if you select this access mode." }) }),
        /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.browser-mode-title", children: /* @__PURE__ */ jsx(Text, { weight: "medium", children: "Browser access mode:" }) }),
        /* @__PURE__ */ jsx("p", { children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.browser-mode-description", children: "All requests will be made from the browser directly to the data source and may be subject to Cross-Origin Resource Sharing (CORS) requirements. The URL needs to be accessible from the browser if you select this access mode." }) })
      ]
    }
  );
};
const LABEL_WIDTH = 26;
const DataSourceHttpSettings = (props) => {
  var _a;
  const {
    defaultUrl,
    dataSourceConfig,
    onChange,
    showAccessOptions,
    sigV4AuthToggleEnabled,
    showForwardOAuthIdentityOption,
    azureAuthSettings,
    renderSigV4Editor,
    secureSocksDSProxyEnabled,
    urlLabel,
    urlDocs
  } = props;
  const ACCESS_OPTIONS = useMemo(
    () => [
      {
        label: t("grafana-ui.data-source-http-settings.server-mode-label", "Server (default)"),
        value: "proxy"
      },
      {
        label: t("grafana-ui.data-source-http-settings.browser-mode-label", "Browser"),
        value: "direct"
      }
    ],
    []
  );
  const DEFAULT_ACCESS_OPTION = useMemo(() => ACCESS_OPTIONS[0], [ACCESS_OPTIONS]);
  const [isAccessHelpVisible, setIsAccessHelpVisible] = useState(false);
  const [azureAuthEnabled, setAzureAuthEnabled] = useState(false);
  const theme = useTheme2();
  let urlTooltip;
  const onSettingsChange = useCallback(
    (change) => {
      const isAzureAuthEnabled = (azureAuthSettings == null ? void 0 : azureAuthSettings.azureAuthSupported) && azureAuthSettings.getAzureAuthEnabled(dataSourceConfig) || false;
      setAzureAuthEnabled(isAzureAuthEnabled);
      if (isAzureAuthEnabled) {
        const tmpOauthPassThru = dataSourceConfig.jsonData.oauthPassThru !== void 0 ? dataSourceConfig.jsonData.oauthPassThru : false;
        change = {
          ...change,
          jsonData: {
            ...dataSourceConfig.jsonData,
            oauthPassThru: isAzureAuthEnabled ? false : tmpOauthPassThru
          }
        };
      }
      onChange({
        ...dataSourceConfig,
        ...change
      });
    },
    [azureAuthSettings, dataSourceConfig, onChange]
  );
  switch (dataSourceConfig.access) {
    case "direct":
      urlTooltip = /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsxs(Trans, { i18nKey: "grafana-ui.data-source-http-settings.direct-url-tooltip", children: [
          "Your access method is ",
          /* @__PURE__ */ jsx("em", { children: "Browser" }),
          ", this means the URL needs to be accessible from the browser."
        ] }),
        urlDocs
      ] });
      break;
    case "proxy":
      urlTooltip = /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsxs(Trans, { i18nKey: "grafana-ui.data-source-http-settings.proxy-url-tooltip", children: [
          "Your access method is ",
          /* @__PURE__ */ jsx("em", { children: "Server" }),
          ", this means the URL needs to be accessible from the grafana backend/server."
        ] }),
        urlDocs
      ] });
      break;
    default:
      urlTooltip = /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.default-url-tooltip", children: "Specify a complete HTTP URL (for example http://your_server:8080)" }),
        urlDocs
      ] });
  }
  const isValidUrl = /^(ftp|http|https):\/\/(\w+:{0,1}\w*@)?(\S+)(:[0-9]+)?(\/|\/([\w#!:.?+=&%@!\-\/]))?$/.test(
    dataSourceConfig.url
  );
  const gridLayout = css({
    display: "grid",
    gridTemplateColumns: "auto 1fr",
    gap: theme.spacing(0.5)
  });
  const fromFieldId = useId();
  return /* @__PURE__ */ jsxs(Stack, { direction: "column", gap: 5, children: [
    /* @__PURE__ */ jsxs("section", { children: [
      /* @__PURE__ */ jsx("h3", { className: "page-heading", children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.heading", children: "HTTP" }) }),
      /* @__PURE__ */ jsx(
        Field,
        {
          label: urlLabel != null ? urlLabel : "URL",
          description: urlTooltip,
          invalid: !isValidUrl,
          error: !isValidUrl && t("grafana-ui.data-source-http-settings.invalid-url-error", "Invalid URL"),
          disabled: dataSourceConfig.readOnly,
          children: /* @__PURE__ */ jsx(
            Input,
            {
              id: fromFieldId,
              width: 40,
              placeholder: defaultUrl,
              value: dataSourceConfig.url,
              "data-testid": selectors.components.DataSource.DataSourceHttpSettings.urlInput,
              onChange: (event) => onSettingsChange({ url: event.currentTarget.value })
            }
          )
        }
      ),
      showAccessOptions && /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx(
          Field,
          {
            label: t("grafana-ui.data-source-http-settings.access-label", "Access"),
            disabled: dataSourceConfig.readOnly,
            children: /* @__PURE__ */ jsxs(Stack, { direction: "row", gap: 0.5, children: [
              /* @__PURE__ */ jsx(
                RadioButtonGroup,
                {
                  "aria-label": t("grafana-ui.data-source-http-settings.access-label", "Access"),
                  options: ACCESS_OPTIONS,
                  value: ((_a = ACCESS_OPTIONS.find((o) => o.value === dataSourceConfig.access)) == null ? void 0 : _a.value) || DEFAULT_ACCESS_OPTION.value,
                  onChange: (selectedValue) => onSettingsChange({ access: selectedValue })
                }
              ),
              /* @__PURE__ */ jsx(
                Button,
                {
                  type: "button",
                  variant: "secondary",
                  size: "md",
                  fill: "outline",
                  onClick: () => setIsAccessHelpVisible((isVisible) => !isVisible),
                  "aria-expanded": isAccessHelpVisible,
                  "aria-controls": ACCESS_HELP_ID,
                  children: /* @__PURE__ */ jsxs(Trans, { i18nKey: "grafana-ui.data-source-http-settings.access-help", children: [
                    "Help\xA0",
                    /* @__PURE__ */ jsx(Icon, { name: isAccessHelpVisible ? "angle-down" : "angle-right" })
                  ] })
                }
              )
            ] })
          }
        ),
        isAccessHelpVisible && /* @__PURE__ */ jsx(HttpAccessHelp, {})
      ] }),
      dataSourceConfig.access === "proxy" && /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx(
          Field,
          {
            label: t("grafana-ui.data-source-http-settings.allowed-cookies", "Allowed cookies"),
            description: t(
              "grafana-ui.data-source-http-settings.allowed-cookies-description",
              "Grafana proxy deletes forwarded cookies by default. Specify cookies by name that should be forwarded to the data source."
            ),
            children: /* @__PURE__ */ jsx(
              TagsInput,
              {
                tags: dataSourceConfig.jsonData.keepCookies,
                width: 40,
                onChange: (cookies) => onSettingsChange({ jsonData: { ...dataSourceConfig.jsonData, keepCookies: cookies } }),
                disabled: dataSourceConfig.readOnly
              }
            )
          }
        ),
        /* @__PURE__ */ jsx(
          Field,
          {
            label: t("grafana-ui.data-source-http-settings.timeout-label", "Timeout"),
            description: t(
              "grafana-ui.data-source-http-settings.timeout-description",
              "HTTP request timeout in seconds"
            ),
            disabled: dataSourceConfig.readOnly,
            children: /* @__PURE__ */ jsx(
              Input,
              {
                type: "number",
                width: 40,
                placeholder: t("grafana-ui.data-source-http-settings.timeout-placeholder", "Timeout in seconds"),
                value: dataSourceConfig.jsonData.timeout,
                onChange: (event) => {
                  onSettingsChange({
                    jsonData: { ...dataSourceConfig.jsonData, timeout: parseInt(event.currentTarget.value, 10) }
                  });
                }
              }
            )
          }
        )
      ] })
    ] }),
    /* @__PURE__ */ jsxs("section", { children: [
      /* @__PURE__ */ jsx("h3", { className: "page-heading", children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.auth", children: "Auth" }) }),
      /* @__PURE__ */ jsxs(Stack, { direction: "column", gap: 4, children: [
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsxs("div", { className: gridLayout, children: [
            /* @__PURE__ */ jsx(
              InlineField,
              {
                label: t("grafana-ui.data-source-http-settings.basic-auth-label", "Basic auth"),
                labelWidth: LABEL_WIDTH,
                disabled: dataSourceConfig.readOnly,
                children: /* @__PURE__ */ jsx(
                  InlineSwitch,
                  {
                    id: "http-settings-basic-auth",
                    value: dataSourceConfig.basicAuth,
                    onChange: (event) => {
                      onSettingsChange({ basicAuth: event.currentTarget.checked });
                    }
                  }
                )
              }
            ),
            /* @__PURE__ */ jsx(
              InlineField,
              {
                label: t("grafana-ui.data-source-http-settings.with-credentials-label", "With Credentials"),
                tooltip: t(
                  "grafana-ui.data-source-http-settings.with-credentials-tooltip",
                  "Whether credentials such as cookies or auth headers should be sent with cross-site requests."
                ),
                labelWidth: LABEL_WIDTH,
                disabled: dataSourceConfig.readOnly,
                children: /* @__PURE__ */ jsx(
                  InlineSwitch,
                  {
                    id: "http-settings-with-credentials",
                    value: dataSourceConfig.withCredentials,
                    onChange: (event) => {
                      onSettingsChange({ withCredentials: event.currentTarget.checked });
                    }
                  }
                )
              }
            ),
            (azureAuthSettings == null ? void 0 : azureAuthSettings.azureAuthSupported) && /* @__PURE__ */ jsx(
              InlineField,
              {
                label: t("grafana-ui.data-source-http-settings.azure-auth-label", "Azure Authentication"),
                tooltip: t(
                  "grafana-ui.data-source-http-settings.azure-auth-tooltip",
                  "Use Azure authentication for Azure endpoint."
                ),
                labelWidth: LABEL_WIDTH,
                disabled: dataSourceConfig.readOnly,
                children: /* @__PURE__ */ jsx(
                  InlineSwitch,
                  {
                    id: "http-settings-azure-auth",
                    value: azureAuthEnabled,
                    onChange: (event) => {
                      onSettingsChange(
                        azureAuthSettings.setAzureAuthEnabled(dataSourceConfig, event.currentTarget.checked)
                      );
                    }
                  }
                )
              }
            ),
            sigV4AuthToggleEnabled && /* @__PURE__ */ jsx(
              InlineField,
              {
                label: t("grafana-ui.data-source-http-settings.sigv4-auth-label", "SigV4 auth"),
                labelWidth: LABEL_WIDTH,
                disabled: dataSourceConfig.readOnly,
                children: /* @__PURE__ */ jsx(
                  InlineSwitch,
                  {
                    id: "http-settings-sigv4-auth",
                    value: dataSourceConfig.jsonData.sigV4Auth || false,
                    onChange: (event) => {
                      onSettingsChange({
                        jsonData: { ...dataSourceConfig.jsonData, sigV4Auth: event.currentTarget.checked }
                      });
                    }
                  }
                )
              }
            )
          ] }),
          dataSourceConfig.access === "proxy" && /* @__PURE__ */ jsx(
            HttpProxySettings,
            {
              dataSourceConfig,
              onChange: (jsonData) => onSettingsChange({ jsonData }),
              showForwardOAuthIdentityOption: azureAuthEnabled ? false : showForwardOAuthIdentityOption
            }
          )
        ] }),
        dataSourceConfig.basicAuth && /* @__PURE__ */ jsxs("section", { children: [
          /* @__PURE__ */ jsx(Text, { variant: "h6", element: "h4", children: /* @__PURE__ */ jsx(Trans, { i18nKey: "grafana-ui.data-source-http-settings.basic-auth", children: "Basic Auth Details" }) }),
          /* @__PURE__ */ jsx(BasicAuthSettings, { ...props })
        ] }),
        (azureAuthSettings == null ? void 0 : azureAuthSettings.azureAuthSupported) && azureAuthEnabled && azureAuthSettings.azureSettingsUI && /* @__PURE__ */ jsx(azureAuthSettings.azureSettingsUI, { dataSourceConfig, onChange }),
        dataSourceConfig.jsonData.sigV4Auth && sigV4AuthToggleEnabled && renderSigV4Editor,
        (dataSourceConfig.jsonData.tlsAuth || dataSourceConfig.jsonData.tlsAuthWithCACert) && /* @__PURE__ */ jsx(TLSAuthSettings, { dataSourceConfig, onChange }),
        dataSourceConfig.access === "proxy" && /* @__PURE__ */ jsx(CustomHeadersSettings, { dataSourceConfig, onChange })
      ] })
    ] }),
    secureSocksDSProxyEnabled && /* @__PURE__ */ jsx(SecureSocksProxySettings, { options: dataSourceConfig, onOptionsChange: onChange })
  ] });
};

export { DataSourceHttpSettings };
//# sourceMappingURL=DataSourceHttpSettings.mjs.map
