import { screen } from '@testing-library/react';
import { render } from 'test/test-utils';

import { SceneTimeRange, SceneVariableSet, TestVariable } from '@grafana/scenes';

import { DashboardScene } from '../../../features/dashboard-scene/scene/DashboardScene';
import { ReportBaseV2 } from '../../types';

import ReportForm from './ReportForm';

jest.mock('app/core/core', () => {
  return {
    contextSrv: {
      hasPermission: () => true,
    },
  };
});

jest.mock('app/features/dashboard/api/dashboard_api', () => ({
  getDashboardAPI: () => ({
    getDashboardDTO: jest.fn().mockResolvedValue({
      dashboard: {
        uid: 'test-dashboard',
        title: 'Test Dashboard',
      },
      meta: {
        folderTitle: 'Test Folder',
        folderUid: 'test-folder',
      },
    }),
  }),
}));

jest.mock('app/core/services/backend_srv', () => ({
  backendSrv: {
    search: jest.fn().mockResolvedValue([]),
  },
}));

function setup({
  report,
  dashboardProps,
}: {
  report?: Partial<ReportBaseV2>;
  dashboardProps?: { dashboard?: DashboardScene };
} = {}) {
  const varA = new TestVariable({ name: 'A', query: 'A.*', value: 'A.AA', text: '', options: [], delayMs: 0 });
  const testDashboard = new DashboardScene({
    uid: 'test-dashboard',
    title: 'Test Dashboard',
    $timeRange: new SceneTimeRange({
      timeZone: 'browser',
    }),
    $variables: new SceneVariableSet({ variables: [varA] }),
  });

  const updatedProps = {
    report: {
      title: 'Report name test',
      dashboards: [
        {
          uid: testDashboard.state.uid,
          key: testDashboard.state.key,
        },
      ],
      ...report,
    },
    dashboard: testDashboard,
    ...dashboardProps,
  };

  return render(<ReportForm {...updatedProps} />);
}

describe('ReportForm', () => {
  it('should render with sections and action buttons', async () => {
    setup();

    expect(screen.getByRole('textbox', { name: /report name/i })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: /dashboard/i })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: 'Schedule' })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: /email settings/i })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: /recipients/i })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: /attachments/i })).toBeInTheDocument();

    // Check for action buttons
    expect(screen.getByRole('button', { name: /schedule report/i })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: /save draft/i })).toBeInTheDocument();
    expect(screen.getByRole('button', { name: /send preview/i })).toBeInTheDocument();
  });

  it('should show validation error when submitting without required fields', async () => {
    const { user } = setup({
      report: { title: undefined, dashboards: [{ uid: undefined }] },
      dashboardProps: { dashboard: undefined },
    });

    await user.click(screen.getByRole('button', { name: /dashboard/i }));
    await user.click(screen.getByRole('button', { name: /schedule report/i }));

    expect(await screen.findByText(/report name is required/i)).toBeInTheDocument();
    expect(await screen.findByText(/dashboard is required/i)).toBeInTheDocument();
    expect(await screen.findByText(/recipients are required/i)).toBeInTheDocument();
  });

  it('should handle form submission with valid data', async () => {
    const { user } = setup();

    await user.type(screen.getByRole('textbox', { name: /report name/i }), 'Test Report');

    const recipientInput = screen.getByPlaceholderText(/Type in the recipients email addresses/i);
    await user.type(recipientInput, 'test@example.com');
    await user.keyboard('{Enter}');

    await user.click(screen.getByRole('button', { name: /schedule report/i }));

    // Verify no validation errors are shown
    expect(screen.queryByText(/report name is required/i)).not.toBeInTheDocument();
    expect(screen.queryByText(/recipients are required/i)).not.toBeInTheDocument();
  });

  describe('Dashboards functionality', () => {
    it('should allow adding multiple dashboards', async () => {
      const { user } = setup();

      await user.click(screen.getByRole('button', { name: /dashboard/i }));

      // Add second dashboard
      await user.click(screen.getByRole('button', { name: /add dashboard/i }));
      const sourceDashboardInput = screen.getAllByText(/source dashboard/i);
      expect(sourceDashboardInput).toHaveLength(2);
    });

    it('should allow removing dashboards', async () => {
      const { user } = setup();

      await user.click(screen.getByRole('button', { name: /dashboard/i }));
      await user.click(screen.getByRole('button', { name: /add dashboard/i }));

      const removeButtons = screen.getAllByRole('button', { name: /delete this dashboard/i });
      await user.click(removeButtons[1]);

      // Verify the dashboard was removed
      const sourceDashboardInput = screen.getAllByText(/source dashboard/i);
      expect(sourceDashboardInput).toHaveLength(1);
    });

    it('should handle dashboard with variables correctly', async () => {
      const varA = new TestVariable({ name: 'A', query: 'A.*', value: 'A.AA', text: '', options: [], delayMs: 0 });
      const dashboardWithVars = new DashboardScene({
        uid: 'dashboard-with-vars',
        title: 'Dashboard with Variables',
        $timeRange: new SceneTimeRange({
          timeZone: 'browser',
        }),
        $variables: new SceneVariableSet({ variables: [varA] }),
      });

      const { user } = setup({ dashboardProps: { dashboard: dashboardWithVars } });

      await user.click(screen.getByRole('button', { name: /dashboard/i }));

      // Verify variables section is present
      expect(screen.getByText(/customize template variables/i)).toBeInTheDocument();
    });

    it('should handle dashboard without variables correctly', async () => {
      const dashboardWithoutVars = new DashboardScene({
        uid: 'dashboard-without-vars',
        title: 'Dashboard without Variables',
        $timeRange: new SceneTimeRange({
          timeZone: 'browser',
        }),
      });

      const { user } = setup({ dashboardProps: { dashboard: dashboardWithoutVars } });

      await user.click(screen.getByRole('button', { name: /dashboard/i }));

      // Verify no variables section is present
      expect(screen.queryByText(/customize template variables/i)).not.toBeInTheDocument();
    });
  });
});
