import { Registry } from '@grafana/data';
import { config } from '@grafana/runtime';
import { prometheusRegularEscape } from '../../datasource.mjs';
import { utf8Support, isValidLegacyName } from '../../utf8_support.mjs';
import { PromVisualQueryOperationCategory } from '../types.mjs';

"use strict";
class LokiAndPromQueryModellerBase {
  constructor(getOperations) {
    this.categories = [];
    this.operationsRegistry = new Registry(getOperations);
  }
  setOperationCategories(categories) {
    this.categories = categories;
  }
  getOperationsForCategory(category) {
    return this.operationsRegistry.list().filter((op) => op.category === category && !op.hideFromList);
  }
  getAlternativeOperations(key) {
    return this.operationsRegistry.list().filter((op) => op.alternativesKey && op.alternativesKey === key);
  }
  getCategories() {
    return this.categories;
  }
  getOperationDef(id) {
    return this.operationsRegistry.getIfExists(id);
  }
  renderOperations(queryString, operations) {
    for (const operation of operations) {
      const def = this.operationsRegistry.getIfExists(operation.id);
      if (!def) {
        throw new Error(`Could not find operation ${operation.id} in the registry`);
      }
      queryString = def.renderer(operation, def, queryString);
    }
    return queryString;
  }
  renderBinaryQueries(queryString, binaryQueries) {
    if (binaryQueries) {
      for (const binQuery of binaryQueries) {
        queryString = `${this.renderBinaryQuery(queryString, binQuery)}`;
      }
    }
    return queryString;
  }
  renderBinaryQuery(leftOperand, binaryQuery) {
    let result = leftOperand + ` ${binaryQuery.operator} `;
    if (binaryQuery.vectorMatches) {
      result += `${binaryQuery.vectorMatchesType}(${binaryQuery.vectorMatches}) `;
    }
    return result + this.renderQuery(binaryQuery.query, true);
  }
  renderLabels(labels) {
    if (labels.length === 0) {
      return "";
    }
    let expr = "{";
    for (const filter of labels) {
      if (expr !== "{") {
        expr += ", ";
      }
      let labelValue = filter.value;
      const usingRegexOperator = filter.op === "=~" || filter.op === "!~";
      if (config.featureToggles.prometheusSpecialCharsInLabelValues && !usingRegexOperator) {
        labelValue = prometheusRegularEscape(labelValue);
      }
      expr += `${utf8Support(filter.label)}${filter.op}"${labelValue}"`;
    }
    return expr + `}`;
  }
  renderQuery(query, nested) {
    var _a, _b;
    let queryString = "";
    const labels = this.renderLabels(query.labels);
    if (query.metric) {
      if (isValidLegacyName(query.metric)) {
        queryString = `${query.metric}${labels}`;
      } else {
        queryString = `{"${query.metric}"${labels.length > 0 ? `, ${labels.substring(1)}` : `}`}`;
      }
    } else {
      queryString = labels;
    }
    queryString = this.renderOperations(queryString, query.operations);
    if (!nested && this.hasBinaryOp(query) && Boolean((_a = query.binaryQueries) == null ? void 0 : _a.length)) {
      queryString = `(${queryString})`;
    }
    queryString = this.renderBinaryQueries(queryString, query.binaryQueries);
    if (nested && (this.hasBinaryOp(query) || Boolean((_b = query.binaryQueries) == null ? void 0 : _b.length))) {
      queryString = `(${queryString})`;
    }
    return queryString;
  }
  hasBinaryOp(query) {
    return query.operations.find((op) => {
      const def = this.getOperationDef(op.id);
      return (def == null ? void 0 : def.category) === PromVisualQueryOperationCategory.BinaryOps;
    }) !== void 0;
  }
}

export { LokiAndPromQueryModellerBase };
//# sourceMappingURL=LokiAndPromQueryModellerBase.mjs.map
