import { jsx, jsxs } from 'react/jsx-runtime';
import { Component } from 'react';
import { getDefaultTimeRange } from '@grafana/data';
import { LoadingPlaceholder, Stack, withTheme2 } from '@grafana/ui';
import { LabelSelector } from './LabelSelector.mjs';
import { MetricSelector } from './MetricSelector.mjs';
import { SelectorActions } from './SelectorActions.mjs';
import { ValueSelector } from './ValueSelector.mjs';
import { buildSelector, facetLabels } from './selectorBuilder.mjs';
import { getStyles } from './styles.mjs';
import { LAST_USED_LABELS_KEY, METRIC_LABEL, EMPTY_SELECTOR, DEFAULT_SERIES_LIMIT } from './types.mjs';

"use strict";
class UnthemedPrometheusMetricsBrowser extends Component {
  constructor() {
    super(...arguments);
    this.state = {
      labels: [],
      labelSearchTerm: "",
      metricSearchTerm: "",
      status: "Ready",
      error: "",
      validationStatus: "",
      valueSearchTerm: ""
    };
    this.onChangeLabelSearch = (event) => {
      this.setState({ labelSearchTerm: event.target.value });
    };
    this.onChangeMetricSearch = (event) => {
      this.setState({ metricSearchTerm: event.target.value });
    };
    this.onChangeSeriesLimit = (event) => {
      this.setState({ seriesLimit: event.target.value.trim() });
    };
    this.onChangeValueSearch = (event) => {
      this.setState({ valueSearchTerm: event.target.value });
    };
    this.onClickRunQuery = () => {
      const selector = buildSelector(this.state.labels);
      this.props.onChange(selector);
    };
    this.onClickRunRateQuery = () => {
      const selector = buildSelector(this.state.labels);
      const query = `rate(${selector}[$__rate_interval])`;
      this.props.onChange(query);
    };
    this.onClickClear = () => {
      this.setState((state) => {
        const labels = state.labels.map((label) => ({
          ...label,
          values: void 0,
          selected: false,
          loading: false,
          hidden: false,
          facets: void 0
        }));
        return {
          labels,
          labelSearchTerm: "",
          metricSearchTerm: "",
          status: "",
          error: "",
          validationStatus: "",
          valueSearchTerm: ""
        };
      });
      localStorage.removeItem(LAST_USED_LABELS_KEY);
      this.fetchValues(METRIC_LABEL, EMPTY_SELECTOR);
    };
    this.onClickLabel = (name, value, event) => {
      const label = this.state.labels.find((l) => l.name === name);
      if (!label) {
        return;
      }
      const selected = !label.selected;
      let nextValue = { selected };
      if (label.values && !selected) {
        const values = label.values.map((value2) => ({ ...value2, selected: false }));
        nextValue = { ...nextValue, facets: 0, values };
      }
      this.setState({ labelSearchTerm: "" });
      this.updateLabelState(name, nextValue, "", () => this.doFacettingForLabel(name));
    };
    this.onClickValue = (name, value, event) => {
      const label = this.state.labels.find((l) => l.name === name);
      if (!label || !label.values) {
        return;
      }
      this.setState({ labelSearchTerm: "" });
      const values = label.values.map((v) => ({ ...v, selected: v.name === value ? !v.selected : v.selected }));
      this.updateLabelState(name, { values }, "", () => this.doFacetting(name));
    };
    this.onClickMetric = (name, value, event) => {
      const label = this.state.labels.find((l) => l.name === name);
      if (!label || !label.values) {
        return;
      }
      this.setState({ metricSearchTerm: "" });
      const values = label.values.map((v) => ({
        ...v,
        selected: v.name === value || v.selected ? !v.selected : v.selected
      }));
      const selected = values.some((v) => v.selected);
      this.updateLabelState(name, { selected, values }, "", () => this.doFacetting(name));
    };
    this.onClickValidate = () => {
      const selector = buildSelector(this.state.labels);
      this.validateSelector(selector);
    };
    this.doFacetting = (lastFacetted) => {
      const selector = buildSelector(this.state.labels);
      if (selector === EMPTY_SELECTOR) {
        const labels = this.state.labels.map((label) => {
          return { ...label, facets: 0, values: void 0, hidden: false };
        });
        this.setState({ labels }, () => {
          this.state.labels.forEach(
            (label) => (label.selected || label.name === METRIC_LABEL) && this.fetchValues(label.name, selector)
          );
        });
      } else {
        this.fetchSeries(selector, lastFacetted);
      }
    };
  }
  updateLabelState(name, updatedFields, status = "", cb) {
    this.setState((state) => {
      const labels = state.labels.map((label) => {
        if (label.name === name) {
          return { ...label, ...updatedFields };
        }
        return label;
      });
      const error = status ? "" : state.error;
      return { labels, status, error, validationStatus: "" };
    }, cb);
  }
  componentDidMount() {
    var _a, _b;
    const { languageProvider } = this.props;
    if (languageProvider) {
      const selectedLabels = (_b = JSON.parse((_a = localStorage.getItem(LAST_USED_LABELS_KEY)) != null ? _a : `[]`)) != null ? _b : [];
      languageProvider.start(this.props.timeRange).then(() => {
        let rawLabels = languageProvider.getLabelKeys();
        this.fetchValues(METRIC_LABEL, EMPTY_SELECTOR);
        const labels = rawLabels.map((label, i, arr) => ({
          name: label,
          selected: selectedLabels.includes(label),
          loading: false
        }));
        this.setState({ labels }, () => {
          this.state.labels.forEach((label) => {
            if (label.selected) {
              this.fetchValues(label.name, EMPTY_SELECTOR);
            }
          });
        });
      });
    }
  }
  doFacettingForLabel(name) {
    const label = this.state.labels.find((l) => l.name === name);
    if (!label) {
      return;
    }
    const selectedLabels = this.state.labels.filter((label2) => label2.selected).map((label2) => label2.name);
    localStorage.setItem(LAST_USED_LABELS_KEY, JSON.stringify(selectedLabels));
    if (label.selected) {
      if (!label.values) {
        this.fetchValues(name, buildSelector(this.state.labels));
      }
    } else {
      this.doFacetting();
    }
  }
  async fetchValues(name, selector) {
    var _a;
    const { languageProvider } = this.props;
    this.updateLabelState(name, { loading: true }, `Fetching values for ${name}`);
    try {
      let rawValues = await languageProvider.getLabelValues((_a = this.props.timeRange) != null ? _a : getDefaultTimeRange(), name);
      if (selector !== buildSelector(this.state.labels)) {
        this.updateLabelState(name, { loading: false });
        return;
      }
      const values = [];
      const { metricsMetadata } = languageProvider;
      for (const labelValue of rawValues) {
        const value = { name: labelValue };
        if (name === METRIC_LABEL && metricsMetadata) {
          const meta = metricsMetadata[labelValue];
          if (meta) {
            value.details = `(${meta.type}) ${meta.help}`;
          }
        }
        values.push(value);
      }
      this.updateLabelState(name, { values, loading: false });
    } catch (error) {
      console.error(error);
    }
  }
  async fetchSeries(selector, lastFacetted) {
    var _a;
    const { languageProvider } = this.props;
    if (lastFacetted) {
      this.updateLabelState(lastFacetted, { loading: true }, `Facetting labels for ${selector}`);
    }
    try {
      const possibleLabels = await languageProvider.fetchSeriesLabels(
        (_a = this.props.timeRange) != null ? _a : getDefaultTimeRange(),
        selector,
        true,
        this.state.seriesLimit
      );
      if (selector !== buildSelector(this.state.labels)) {
        if (lastFacetted) {
          this.updateLabelState(lastFacetted, { loading: false });
        }
        return;
      }
      if (Object.keys(possibleLabels).length === 0) {
        this.setState({ error: `Empty results, no matching label for ${selector}` });
        return;
      }
      const labels = facetLabels(this.state.labels, possibleLabels, lastFacetted);
      this.setState({ labels, error: "" });
      if (lastFacetted) {
        this.updateLabelState(lastFacetted, { loading: false });
      }
    } catch (error) {
      console.error(error);
    }
  }
  async validateSelector(selector) {
    var _a;
    const { languageProvider } = this.props;
    this.setState({ validationStatus: `Validating selector ${selector}`, error: "" });
    const streams = await languageProvider.fetchSeries((_a = this.props.timeRange) != null ? _a : getDefaultTimeRange(), selector);
    this.setState({ validationStatus: `Selector is valid (${streams.length} series found)` });
  }
  render() {
    var _a, _b;
    const { theme } = this.props;
    const { labels, labelSearchTerm, metricSearchTerm, status, error, validationStatus, valueSearchTerm } = this.state;
    const styles = getStyles(theme);
    if (labels.length === 0) {
      return /* @__PURE__ */ jsx("div", { className: styles.wrapper, children: /* @__PURE__ */ jsx(LoadingPlaceholder, { text: "Loading labels..." }) });
    }
    let metrics = labels.find((label) => label.name === METRIC_LABEL);
    if (metrics && metricSearchTerm) {
      metrics = {
        ...metrics,
        values: (_a = metrics.values) == null ? void 0 : _a.filter((value) => value.selected || value.name.includes(metricSearchTerm))
      };
    }
    let nonMetricLabels = labels.filter((label) => !label.hidden && label.name !== METRIC_LABEL);
    if (labelSearchTerm) {
      nonMetricLabels = nonMetricLabels.filter((label) => label.selected || label.name.includes(labelSearchTerm));
    }
    let selectedLabels = nonMetricLabels.filter((label) => label.selected && label.values);
    if (valueSearchTerm) {
      selectedLabels = selectedLabels.map((label) => {
        var _a2;
        return {
          ...label,
          values: (_a2 = label.values) == null ? void 0 : _a2.filter((value) => value.selected || value.name.includes(valueSearchTerm))
        };
      });
    }
    const selector = buildSelector(this.state.labels);
    const empty = selector === EMPTY_SELECTOR;
    return /* @__PURE__ */ jsxs("div", { className: styles.wrapper, children: [
      /* @__PURE__ */ jsxs(Stack, { gap: 3, children: [
        /* @__PURE__ */ jsx(
          MetricSelector,
          {
            metrics,
            metricSearchTerm,
            seriesLimit: (_b = this.state.seriesLimit) != null ? _b : DEFAULT_SERIES_LIMIT,
            onChangeMetricSearch: this.onChangeMetricSearch,
            onChangeSeriesLimit: this.onChangeSeriesLimit,
            onClickMetric: this.onClickMetric,
            styles
          }
        ),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx(
            LabelSelector,
            {
              nonMetricLabels,
              labelSearchTerm,
              onChangeLabelSearch: this.onChangeLabelSearch,
              onClickLabel: this.onClickLabel,
              styles
            }
          ),
          /* @__PURE__ */ jsx(
            ValueSelector,
            {
              selectedLabels,
              valueSearchTerm,
              onChangeValueSearch: this.onChangeValueSearch,
              onClickValue: this.onClickValue,
              onClickLabel: this.onClickLabel,
              styles
            }
          )
        ] })
      ] }),
      /* @__PURE__ */ jsx(
        SelectorActions,
        {
          selector,
          validationStatus,
          status,
          error,
          empty,
          onClickRunQuery: this.onClickRunQuery,
          onClickRunRateQuery: this.onClickRunRateQuery,
          onClickValidate: this.onClickValidate,
          onClickClear: this.onClickClear,
          styles
        }
      )
    ] });
  }
}
const PrometheusMetricsBrowser = withTheme2(UnthemedPrometheusMetricsBrowser);

export { PrometheusMetricsBrowser, UnthemedPrometheusMetricsBrowser };
//# sourceMappingURL=PrometheusMetricsBrowser.mjs.map
