import { getActiveThreshold } from '../field/thresholds.mjs';
import { stringToJsRegex } from '../text/string.mjs';
import { MappingType, SpecialValueMatch } from '../types/valueMapping.mjs';

"use strict";
function getValueMappingResult(valueMappings, value) {
  var _a, _b;
  for (const vm of valueMappings) {
    switch (vm.type) {
      case MappingType.ValueToText:
        if (value == null) {
          continue;
        }
        const result = vm.options[value];
        if (result) {
          return result;
        }
        break;
      case MappingType.RangeToText:
        if (value == null) {
          continue;
        }
        const valueAsNumber = parseFloat(value);
        if (isNaN(valueAsNumber)) {
          continue;
        }
        const from = (_a = vm.options.from) != null ? _a : -Infinity;
        const isNumFrom = !isNaN(from);
        if (isNumFrom && valueAsNumber < from) {
          continue;
        }
        const to = (_b = vm.options.to) != null ? _b : Infinity;
        const isNumTo = !isNaN(to);
        if (isNumTo && valueAsNumber > to) {
          continue;
        }
        return vm.options.result;
      case MappingType.RegexToText:
        if (value == null) {
          continue;
        }
        if (typeof value !== "string") {
          continue;
        }
        const regex = stringToJsRegex(vm.options.pattern);
        if (value.match(regex)) {
          const res = { ...vm.options.result };
          if (res.text != null) {
            res.text = value.replace(regex, vm.options.result.text || "");
          }
          return res;
        }
      case MappingType.SpecialValue:
        switch (vm.options.match) {
          case SpecialValueMatch.Null: {
            if (value == null) {
              return vm.options.result;
            }
            break;
          }
          case SpecialValueMatch.NaN: {
            if (typeof value === "number" && isNaN(value)) {
              return vm.options.result;
            }
            break;
          }
          case SpecialValueMatch.NullAndNaN: {
            if (typeof value === "number" && isNaN(value) || value == null) {
              return vm.options.result;
            }
            break;
          }
          case SpecialValueMatch.True: {
            if (value === true || value === "true") {
              return vm.options.result;
            }
            break;
          }
          case SpecialValueMatch.False: {
            if (value === false || value === "false") {
              return vm.options.result;
            }
            break;
          }
          case SpecialValueMatch.Empty: {
            if (value === "") {
              return vm.options.result;
            }
            break;
          }
        }
    }
  }
  return null;
}
function isNumeric(num) {
  return (typeof num === "number" || typeof num === "string" && num.trim() !== "") && !isNaN(num);
}
var LegacyMappingType = /* @__PURE__ */ ((LegacyMappingType2) => {
  LegacyMappingType2[LegacyMappingType2["ValueToText"] = 1] = "ValueToText";
  LegacyMappingType2[LegacyMappingType2["RangeToText"] = 2] = "RangeToText";
  return LegacyMappingType2;
})(LegacyMappingType || {});
function convertOldAngularValueMappings(panel, migratedThresholds) {
  var _a, _b, _c, _d;
  const mappings = [];
  let mappingType = panel.mappingType;
  if (!panel.mappingType) {
    if (panel.valueMaps && panel.valueMaps.length) {
      mappingType = 1;
    } else if (panel.rangeMaps && panel.rangeMaps.length) {
      mappingType = 2;
    }
  }
  if (mappingType === 1) {
    for (let i = 0; i < panel.valueMaps.length; i++) {
      const map = panel.valueMaps[i];
      mappings.push(
        upgradeOldAngularValueMapping(
          {
            ...map,
            id: i,
            // used for order
            type: MappingType.ValueToText
          },
          ((_b = (_a = panel.fieldConfig) == null ? void 0 : _a.defaults) == null ? void 0 : _b.thresholds) || migratedThresholds
        )
      );
    }
  } else if (mappingType === 2) {
    for (let i = 0; i < panel.rangeMaps.length; i++) {
      const map = panel.rangeMaps[i];
      mappings.push(
        upgradeOldAngularValueMapping(
          {
            ...map,
            id: i,
            // used for order
            type: MappingType.RangeToText
          },
          ((_d = (_c = panel.fieldConfig) == null ? void 0 : _c.defaults) == null ? void 0 : _d.thresholds) || migratedThresholds
        )
      );
    }
  }
  return mappings;
}
function upgradeOldAngularValueMapping(old, thresholds) {
  const valueMaps = { type: MappingType.ValueToText, options: {} };
  const newMappings = [];
  let color = void 0;
  const numeric = parseFloat(old.text);
  if (thresholds && !isNaN(numeric)) {
    const level = getActiveThreshold(numeric, thresholds.steps);
    if (level && level.color) {
      color = level.color;
    }
  }
  switch (old.type) {
    case 1 /* ValueToText */:
    case MappingType.ValueToText:
      if (old.value != null) {
        if (old.value === "null") {
          newMappings.push({
            type: MappingType.SpecialValue,
            options: {
              match: SpecialValueMatch.Null,
              result: { text: old.text, color }
            }
          });
        } else {
          valueMaps.options[String(old.value)] = {
            text: old.text,
            color
          };
        }
      }
      break;
    case 2 /* RangeToText */:
    case MappingType.RangeToText:
      if (old.from === "null" || old.to === "null") {
        newMappings.push({
          type: MappingType.SpecialValue,
          options: {
            match: SpecialValueMatch.Null,
            result: { text: old.text, color }
          }
        });
      } else {
        newMappings.push({
          type: MappingType.RangeToText,
          options: {
            from: +old.from,
            to: +old.to,
            result: { text: old.text, color }
          }
        });
      }
      break;
  }
  if (Object.keys(valueMaps.options).length > 0) {
    newMappings.unshift(valueMaps);
  }
  return newMappings[0];
}

export { LegacyMappingType, convertOldAngularValueMappings, getValueMappingResult, isNumeric };
//# sourceMappingURL=valueMappings.mjs.map
