import { screen } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import { render } from 'test/test-utils';

import { SceneTimeRange, SceneVariableSet, TestVariable } from '@grafana/scenes';

import { DashboardScene } from '../../../../features/dashboard-scene/scene/DashboardScene';
import { ReportBaseV2 } from '../../../types';
import ReportForm from '../ReportForm';

jest.mock('app/core/core', () => {
  return {
    contextSrv: {
      hasPermission: () => true,
    },
  };
});

function setup({ report }: { report?: Partial<ReportBaseV2> }) {
  const varA = new TestVariable({ name: 'A', query: 'A.*', value: 'A.AA', text: '', options: [], delayMs: 0 });
  const dashboard = new DashboardScene({
    uid: 'test-dashboard',
    title: 'Test Dashboard',
    $timeRange: new SceneTimeRange({
      timeZone: 'browser',
    }),
    $variables: new SceneVariableSet({ variables: [varA] }),
  });
  const updatedProps = {
    report: {
      title: 'Report name test',
      dashboards: [
        {
          uid: dashboard.state.uid,
          key: dashboard.state.key,
        },
      ],
      ...report,
    },
    dashboard,
  };
  const options = render(<ReportForm {...updatedProps} />);
  return {
    ...options,
    user: userEvent.setup(),
  };
}

describe('EmailConfiguration', () => {
  it('should render', async () => {
    const { user } = setup({});

    const emailConfigurationSection = screen.getByRole('button', { name: /email settings/i });
    expect(emailConfigurationSection).toBeInTheDocument();

    await user.click(emailConfigurationSection);

    expect(await screen.findByRole('textbox', { name: /email subject/i })).toBeInTheDocument();
    expect(await screen.findByRole('textbox', { name: /message/i })).toBeInTheDocument();
    expect(await screen.findByRole('textbox', { name: /reply-to-email address/i })).toBeInTheDocument();
    expect(await screen.findByRole('checkbox', { name: /include dashboard link/i })).toBeInTheDocument();
    expect(await screen.findByRole('checkbox', { name: /embed dashboard image/i })).toBeInTheDocument();
  });
  it('should disable submit button when replyTo email is invalid but should be enabled when it is valid or empty', async () => {
    const { user } = setup({});

    const emailConfigurationSection = screen.getByRole('button', { name: /email settings/i });
    await user.click(emailConfigurationSection);

    const replyToInput = await screen.findByRole('textbox', { name: /reply-to-email address/i });
    await user.type(replyToInput, 'invalid-email');

    expect(screen.getByText('Invalid email')).toBeInTheDocument();

    await user.clear(replyToInput);
    await user.type(replyToInput, 'valid@email.com');

    expect(screen.queryByText('Invalid email')).not.toBeInTheDocument();
  });
});
