import { Observable, of } from 'rxjs';
import { rangeUtil, renderLegendFormat } from '@grafana/data';
import { AnnotationQueryEditor } from './components/AnnotationQueryEditor.mjs';

"use strict";
const ANNOTATION_QUERY_STEP_DEFAULT = "60s";
const PrometheusAnnotationSupport = (ds) => {
  return {
    QueryEditor: AnnotationQueryEditor,
    prepareAnnotation(json) {
      if (!json.target) {
        json.target = {
          expr: "",
          refId: "Anno"
        };
      }
      json.target = {
        ...json.target,
        refId: json.target.refId || json.refId || "Anno",
        expr: json.target.expr || json.expr || "",
        interval: json.target.interval || json.step || ""
      };
      delete json.expr;
      delete json.step;
      return json;
    },
    processEvents(anno, frames) {
      var _a, _b;
      if (!frames.length) {
        return new Observable();
      }
      const { tagKeys = "", titleFormat = "", textFormat = "" } = anno;
      const input = ((_a = frames[0].meta) == null ? void 0 : _a.executedQueryString) || "";
      const regex = /Step:\s*([\d\w]+)/;
      const match = input.match(regex);
      const stepValue = match ? match[1] : null;
      const step = rangeUtil.intervalToSeconds(stepValue || ANNOTATION_QUERY_STEP_DEFAULT) * 1e3;
      const tagKeysArray = tagKeys.split(",");
      const eventList = [];
      for (const frame of frames) {
        if (frame.fields.length === 0) {
          continue;
        }
        const timeField = frame.fields[0];
        const valueField = frame.fields[1];
        const labels = (valueField == null ? void 0 : valueField.labels) || {};
        const tags = Object.keys(labels).filter((label) => tagKeysArray.includes(label)).map((label) => labels[label]);
        const timeValueTuple = [];
        let idx = 0;
        valueField.values.forEach((value) => {
          let timeStampValue;
          let valueValue;
          const time = timeField.values[idx];
          if (anno.useValueForTime) {
            timeStampValue = Math.floor(parseFloat(value));
            valueValue = 1;
          } else {
            timeStampValue = Math.floor(parseFloat(time));
            valueValue = parseFloat(value);
          }
          idx++;
          timeValueTuple.push([timeStampValue, valueValue]);
        });
        const activeValues = timeValueTuple.filter((value) => value[1] > 0);
        const activeValuesTimestamps = activeValues.map((value) => value[0]);
        let latestEvent = null;
        for (const timestamp of activeValuesTimestamps) {
          if (latestEvent && ((_b = latestEvent.timeEnd) != null ? _b : 0) + step >= timestamp) {
            latestEvent.timeEnd = timestamp;
            continue;
          }
          if (latestEvent) {
            eventList.push(latestEvent);
          }
          latestEvent = {
            time: timestamp,
            timeEnd: timestamp,
            annotation: anno,
            title: renderLegendFormat(titleFormat, labels),
            tags,
            text: renderLegendFormat(textFormat, labels)
          };
        }
        if (latestEvent) {
          latestEvent.timeEnd = activeValuesTimestamps[activeValuesTimestamps.length - 1];
          eventList.push(latestEvent);
        }
      }
      return of(eventList);
    }
  };
};

export { PrometheusAnnotationSupport };
//# sourceMappingURL=annotations.mjs.map
