import { map } from 'rxjs/operators';
import { getFieldDisplayName } from '../../field/fieldState.mjs';
import { TransformationApplicabilityLevels } from '../../types/transformations.mjs';
import { reduceField, getFieldTypeForReducer } from '../fieldReducer.mjs';
import { DataTransformerID } from './ids.mjs';
import { findMaxFields } from './utils.mjs';

"use strict";
const MINIMUM_FIELDS_REQUIRED = 2;
var GroupByOperationID = /* @__PURE__ */ ((GroupByOperationID2) => {
  GroupByOperationID2["aggregate"] = "aggregate";
  GroupByOperationID2["groupBy"] = "groupby";
  return GroupByOperationID2;
})(GroupByOperationID || {});
const groupByTransformer = {
  id: DataTransformerID.groupBy,
  name: "Group by",
  description: "Group the data by a field values then process calculations for each group.",
  defaultOptions: {
    fields: {}
  },
  isApplicable: (data) => {
    const maxFields = findMaxFields(data);
    return maxFields >= MINIMUM_FIELDS_REQUIRED ? TransformationApplicabilityLevels.Applicable : TransformationApplicabilityLevels.NotApplicable;
  },
  isApplicableDescription: (data) => {
    const maxFields = findMaxFields(data);
    return `The Group by transformation requires a series with at least ${MINIMUM_FIELDS_REQUIRED} fields to work. The maximum number of fields found on a series is ${maxFields}`;
  },
  /**
   * Return a modified copy of the series. If the transform is not or should not
   * be applied, just return the input series
   */
  operator: (options) => (source) => source.pipe(
    map((data) => {
      var _a;
      const hasValidConfig = Object.keys(options.fields).find(
        (name) => options.fields[name].operation === "groupby" /* groupBy */
      );
      if (!hasValidConfig) {
        return data;
      }
      const processed = [];
      for (const frame of data) {
        const groupByFields = frame.fields.filter((field) => shouldGroupOnField(field, options));
        if (groupByFields.length === 0) {
          continue;
        }
        const valuesByGroupKey = groupValuesByKey(frame, groupByFields);
        const fields = createGroupedFields(groupByFields, valuesByGroupKey);
        for (const field of frame.fields) {
          if (!shouldCalculateField(field, options)) {
            continue;
          }
          const fieldName = getFieldDisplayName(field);
          const aggregations = options.fields[fieldName].aggregations;
          const valuesByAggregation = {};
          valuesByGroupKey.forEach((value) => {
            const fieldWithValuesForGroup = value[fieldName];
            const results = reduceField({
              field: fieldWithValuesForGroup,
              reducers: aggregations
            });
            for (const aggregation of aggregations) {
              if (!Array.isArray(valuesByAggregation[aggregation])) {
                valuesByAggregation[aggregation] = [];
              }
              valuesByAggregation[aggregation].push(results[aggregation]);
            }
          });
          for (const aggregation of aggregations) {
            const aggregationField = {
              name: `${fieldName} (${aggregation})`,
              values: (_a = valuesByAggregation[aggregation]) != null ? _a : [],
              type: getFieldTypeForReducer(aggregation, field.type),
              config: {}
            };
            fields.push(aggregationField);
          }
        }
        processed.push({
          ...frame,
          fields,
          length: valuesByGroupKey.size
        });
      }
      return processed;
    })
  )
};
const shouldGroupOnField = (field, options) => {
  var _a;
  const fieldName = getFieldDisplayName(field);
  return ((_a = options == null ? void 0 : options.fields[fieldName]) == null ? void 0 : _a.operation) === "groupby" /* groupBy */;
};
const shouldCalculateField = (field, options) => {
  var _a;
  const fieldName = getFieldDisplayName(field);
  return ((_a = options == null ? void 0 : options.fields[fieldName]) == null ? void 0 : _a.operation) === "aggregate" /* aggregate */ && Array.isArray(options == null ? void 0 : options.fields[fieldName].aggregations) && (options == null ? void 0 : options.fields[fieldName].aggregations.length) > 0;
};
function groupValuesByKey(frame, groupByFields) {
  var _a;
  const valuesByGroupKey = /* @__PURE__ */ new Map();
  for (let rowIndex = 0; rowIndex < frame.length; rowIndex++) {
    const groupKey = String(groupByFields.map((field) => field.values[rowIndex]));
    const valuesByField = (_a = valuesByGroupKey.get(groupKey)) != null ? _a : {};
    if (!valuesByGroupKey.has(groupKey)) {
      valuesByGroupKey.set(groupKey, valuesByField);
    }
    for (let field of frame.fields) {
      const fieldName = getFieldDisplayName(field);
      if (!valuesByField[fieldName]) {
        valuesByField[fieldName] = {
          name: fieldName,
          type: field.type,
          config: { ...field.config },
          values: []
        };
      }
      valuesByField[fieldName].values.push(field.values[rowIndex]);
    }
  }
  return valuesByGroupKey;
}
function createGroupedFields(groupByFields, valuesByGroupKey) {
  const fields = [];
  for (const field of groupByFields) {
    const values = [];
    const fieldName = getFieldDisplayName(field);
    valuesByGroupKey.forEach((value) => {
      values.push(value[fieldName].values[0]);
    });
    fields.push({
      name: field.name,
      type: field.type,
      config: {
        ...field.config
      },
      values
    });
  }
  return fields;
}

export { GroupByOperationID, createGroupedFields, groupByTransformer, groupValuesByKey };
//# sourceMappingURL=groupBy.mjs.map
