import { jsx, jsxs } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import * as React from 'react';
import { selectors } from '@grafana/e2e-selectors';
import '../../themes/index.mjs';
import { t } from '../../utils/i18n.mjs';
import { Button } from '../Button/Button.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { Box } from '../Layout/Box/Box.mjs';
import { Text } from '../Text/Text.mjs';
import { useTheme2 } from '../../themes/ThemeContext.mjs';

"use strict";
const Alert = React.forwardRef(
  ({
    title,
    onRemove,
    children,
    buttonContent,
    elevated,
    bottomSpacing,
    topSpacing,
    className,
    severity = "error",
    ...restProps
  }, ref) => {
    const theme = useTheme2();
    const hasTitle = Boolean(title);
    const styles = getStyles(theme, severity, hasTitle, elevated, bottomSpacing, topSpacing);
    const rolesBySeverity = {
      error: "alert",
      warning: "alert",
      info: "status",
      success: "status"
    };
    const role = restProps["role"] || rolesBySeverity[severity];
    const ariaLabel = restProps["aria-label"] || title;
    const closeLabel = t("grafana-ui.alert.close-button", "Close alert");
    return /* @__PURE__ */ jsx("div", { ref, className: cx(styles.wrapper, className), role, "aria-label": ariaLabel, ...restProps, children: /* @__PURE__ */ jsxs(
      Box,
      {
        "data-testid": selectors.components.Alert.alertV2(severity),
        display: "flex",
        backgroundColor: severity,
        borderRadius: "default",
        paddingY: 1,
        paddingX: 2,
        borderStyle: "solid",
        borderColor: severity,
        alignItems: "stretch",
        boxShadow: elevated ? "z3" : void 0,
        children: [
          /* @__PURE__ */ jsx(Box, { paddingTop: 1, paddingRight: 2, children: /* @__PURE__ */ jsx("div", { className: styles.icon, children: /* @__PURE__ */ jsx(Icon, { size: "xl", name: getIconFromSeverity(severity) }) }) }),
          /* @__PURE__ */ jsxs(Box, { paddingY: 1, grow: 1, children: [
            /* @__PURE__ */ jsx(Text, { color: "primary", weight: "medium", children: title }),
            children && /* @__PURE__ */ jsx("div", { className: styles.content, children })
          ] }),
          onRemove && !buttonContent && /* @__PURE__ */ jsx("div", { className: styles.close, children: /* @__PURE__ */ jsx(
            Button,
            {
              "aria-label": closeLabel,
              icon: "times",
              onClick: onRemove,
              type: "button",
              fill: "text",
              variant: "secondary"
            }
          ) }),
          onRemove && buttonContent && /* @__PURE__ */ jsx(Box, { marginLeft: 1, display: "flex", alignItems: "center", children: /* @__PURE__ */ jsx(Button, { "aria-label": closeLabel, variant: "secondary", onClick: onRemove, type: "button", children: buttonContent }) })
        ]
      }
    ) });
  }
);
Alert.displayName = "Alert";
const getIconFromSeverity = (severity) => {
  switch (severity) {
    case "error":
      return "exclamation-circle";
    case "warning":
      return "exclamation-triangle";
    case "info":
      return "info-circle";
    case "success":
      return "check";
  }
};
const getStyles = (theme, severity, hasTitle, elevated, bottomSpacing, topSpacing) => {
  const color = theme.colors[severity];
  return {
    wrapper: css({
      flexGrow: 1,
      marginBottom: theme.spacing(bottomSpacing != null ? bottomSpacing : 2),
      marginTop: theme.spacing(topSpacing != null ? topSpacing : 0),
      position: "relative",
      "&:before": {
        content: '""',
        position: "absolute",
        top: 0,
        left: 0,
        bottom: 0,
        right: 0,
        background: theme.colors.background.primary,
        zIndex: -1
      }
    }),
    icon: css({
      color: color.text,
      position: "relative",
      top: "-1px"
    }),
    content: css({
      color: theme.colors.text.primary,
      paddingTop: hasTitle ? theme.spacing(0.5) : 0,
      maxHeight: "50vh",
      overflowY: "auto"
    }),
    close: css({
      position: "relative",
      color: theme.colors.text.secondary,
      background: "none",
      display: "flex",
      top: "-6px",
      right: "-14px"
    })
  };
};

export { Alert, getIconFromSeverity };
//# sourceMappingURL=Alert.mjs.map
