import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { css, cx } from '@emotion/css';
import { useState, useCallback } from 'react';
import { isDataFrame, toLegacyResponseData, getDefaultTimeRange } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { reportInteraction } from '@grafana/runtime';
import { useTheme2, Icon, clearButtonStyles } from '@grafana/ui';
import { getInitHints } from '../query_hints.mjs';
import { makePromiseCancelable, isCancelablePromiseRejection } from './cancelable-promise.mjs';
import { PrometheusMetricsBrowser } from './metrics-browser/PrometheusMetricsBrowser.mjs';
import { MonacoQueryFieldWrapper } from './monaco-query-field/MonacoQueryFieldWrapper.mjs';
import { useMetricsState } from './useMetricsState.mjs';
import { usePromQueryFieldEffects } from './usePromQueryFieldEffects.mjs';

"use strict";
const PromQueryField = (props) => {
  var _a;
  const {
    app,
    datasource,
    datasource: { languageProvider },
    query,
    ExtraFieldElement,
    history = [],
    data,
    range,
    onChange,
    onRunQuery
  } = props;
  const theme = useTheme2();
  const [syntaxLoaded, setSyntaxLoaded] = useState(false);
  const [hint, setHint] = useState(null);
  const [labelBrowserVisible, setLabelBrowserVisible] = useState(false);
  const updateLanguage = useCallback(() => {
    if (languageProvider.metrics) {
      setSyntaxLoaded(true);
    }
  }, [languageProvider]);
  const refreshMetrics = useCallback(
    async (languageProviderInitRef) => {
      if (languageProviderInitRef.current) {
        languageProviderInitRef.current.cancel();
      }
      if (!languageProvider || !range) {
        return;
      }
      try {
        const initialization = makePromiseCancelable(languageProvider.start(range));
        languageProviderInitRef.current = initialization;
        const remainingTasks = await initialization.promise;
        if (Array.isArray(remainingTasks) && remainingTasks.length > 0) {
          await Promise.all(remainingTasks);
        }
        updateLanguage();
      } catch (err) {
        if (isCancelablePromiseRejection(err) && err.isCanceled) {
        } else {
          throw err;
        }
      } finally {
        languageProviderInitRef.current = null;
      }
    },
    [languageProvider, range, updateLanguage]
  );
  const refreshHint = useCallback(() => {
    var _a2, _b;
    const initHints = getInitHints(datasource);
    const initHint = (_a2 = initHints[0]) != null ? _a2 : null;
    if (!((_b = data == null ? void 0 : data.series) == null ? void 0 : _b.length)) {
      setHint(initHint);
      return;
    }
    const result = isDataFrame(data.series[0]) ? data.series.map(toLegacyResponseData) : data.series;
    const queryHints = datasource.getQueryHints(query, result);
    let queryHint = queryHints.length > 0 ? queryHints[0] : null;
    setHint(queryHint != null ? queryHint : initHint);
  }, [data, datasource, query]);
  const onChangeQuery = (value, override) => {
    if (!onChange) {
      return;
    }
    const nextQuery = { ...query, expr: value };
    onChange(nextQuery);
    if (override && onRunQuery) {
      onRunQuery();
    }
  };
  const onChangeLabelBrowser = (selector) => {
    onChangeQuery(selector, true);
    setLabelBrowserVisible(false);
  };
  const onClickChooserButton = () => {
    setLabelBrowserVisible(!labelBrowserVisible);
    reportInteraction("user_grafana_prometheus_metrics_browser_clicked", {
      editorMode: labelBrowserVisible ? "metricViewClosed" : "metricViewOpen",
      app: app != null ? app : ""
    });
  };
  const onClickHintFix = () => {
    var _a2;
    if ((_a2 = hint == null ? void 0 : hint.fix) == null ? void 0 : _a2.action) {
      onChange(datasource.modifyQuery(query, hint.fix.action));
    }
    onRunQuery();
  };
  usePromQueryFieldEffects(languageProvider, range, data == null ? void 0 : data.series, refreshMetrics, refreshHint);
  const { chooserText, buttonDisabled } = useMetricsState(datasource, languageProvider, syntaxLoaded);
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsxs(
      "div",
      {
        className: "gf-form-inline gf-form-inline--xs-view-flex-column flex-grow-1",
        "data-testid": props["data-testid"],
        children: [
          /* @__PURE__ */ jsxs(
            "button",
            {
              className: "gf-form-label query-keyword pointer",
              onClick: onClickChooserButton,
              disabled: buttonDisabled,
              type: "button",
              "data-testid": selectors.components.DataSource.Prometheus.queryEditor.code.metricsBrowser.openButton,
              children: [
                chooserText,
                /* @__PURE__ */ jsx(Icon, { name: labelBrowserVisible ? "angle-down" : "angle-right" })
              ]
            }
          ),
          /* @__PURE__ */ jsx("div", { className: "flex-grow-1 min-width-15", children: /* @__PURE__ */ jsx(
            MonacoQueryFieldWrapper,
            {
              languageProvider,
              history,
              onChange: onChangeQuery,
              onRunQuery,
              initialValue: (_a = query.expr) != null ? _a : "",
              placeholder: "Enter a PromQL query\u2026",
              datasource,
              timeRange: range != null ? range : getDefaultTimeRange()
            }
          ) })
        ]
      }
    ),
    labelBrowserVisible && /* @__PURE__ */ jsx("div", { className: "gf-form", children: /* @__PURE__ */ jsx(
      PrometheusMetricsBrowser,
      {
        languageProvider,
        onChange: onChangeLabelBrowser,
        timeRange: range
      }
    ) }),
    ExtraFieldElement,
    hint ? /* @__PURE__ */ jsx(
      "div",
      {
        className: css({
          flexBasis: "100%"
        }),
        children: /* @__PURE__ */ jsxs("div", { className: "text-warning", children: [
          hint.label,
          " ",
          hint.fix ? /* @__PURE__ */ jsx(
            "button",
            {
              type: "button",
              className: cx(clearButtonStyles(theme), "text-link", "muted"),
              onClick: onClickHintFix,
              children: hint.fix.label
            }
          ) : null
        ] })
      }
    ) : null
  ] });
};

export { PromQueryField };
//# sourceMappingURL=PromQueryField.mjs.map
