import { once } from 'lodash';
import Prism from 'prismjs';
import { LanguageProvider, getDefaultTimeRange, scopeFilterOperatorMap, AbstractLabelOperator } from '@grafana/data';
import { REMOVE_SERIES_LIMIT, DEFAULT_SERIES_LIMIT } from './components/metrics-browser/types.mjs';
import { processHistogramMetrics, processLabels, fixSummariesMetadata, toPromLikeQuery, extractLabelMatchers } from './language_utils.mjs';
import { promqlGrammar } from './promql.mjs';
import { buildVisualQueryFromString } from './querybuilder/parsing.mjs';
import { PrometheusCacheLevel } from './types.mjs';
import { escapeForUtf8Support, isValidLegacyName } from './utf8_support.mjs';

"use strict";
const DEFAULT_KEYS = ["job", "instance"];
const EMPTY_SELECTOR = "{}";
const SUGGESTIONS_LIMIT = 1e4;
const buildCacheHeaders = (durationInSeconds) => {
  return {
    headers: {
      "X-Grafana-Cache": `private, max-age=${durationInSeconds}`
    }
  };
};
function getMetadataString(metric, metadata) {
  if (!metadata[metric]) {
    return void 0;
  }
  const { type, help } = metadata[metric];
  return `${type.toUpperCase()}: ${help}`;
}
function getMetadataHelp(metric, metadata) {
  if (!metadata[metric]) {
    return void 0;
  }
  return metadata[metric].help;
}
function getMetadataType(metric, metadata) {
  if (!metadata[metric]) {
    return void 0;
  }
  return metadata[metric].type;
}
const PREFIX_DELIMITER_REGEX = /(="|!="|=~"|!~"|\{|\[|\(|\+|-|\/|\*|%|\^|\band\b|\bor\b|\bunless\b|==|>=|!=|<=|>|<|=|~|,)/;
const secondsInDay = 86400;
class PromQlLanguageProvider extends LanguageProvider {
  constructor(datasource, initialValues) {
    super();
    this.labelKeys = [];
    this.request = async (url, defaultValue, params = {}, options) => {
      try {
        const res = await this.datasource.metadataRequest(url, params, options);
        return res.data.data;
      } catch (error) {
        if (!isCancelledError(error)) {
          console.error(error);
        }
      }
      return defaultValue;
    };
    this.start = async (timeRange = getDefaultTimeRange()) => {
      if (this.datasource.lookupsDisabled) {
        return [];
      }
      this.metrics = await this.fetchLabelValues(timeRange, "__name__") || [];
      this.histogramMetrics = processHistogramMetrics(this.metrics).sort();
      return Promise.all([this.loadMetricsMetadata(), this.fetchLabels(timeRange)]);
    };
    this.fetchLabelValues = async (range, key) => {
      const params = this.datasource.getAdjustedInterval(range);
      const interpolatedName = this.datasource.interpolateString(key);
      const interpolatedAndEscapedName = escapeForUtf8Support(removeQuotesIfExist(interpolatedName));
      const url = `/api/v1/label/${interpolatedAndEscapedName}/values`;
      const value = await this.request(url, [], params, this.getDefaultCacheHeaders());
      return value != null ? value : [];
    };
    /**
     * Fetches all label keys
     */
    this.fetchLabels = async (timeRange, queries) => {
      let url = "/api/v1/labels";
      const timeParams = this.datasource.getAdjustedInterval(timeRange);
      this.labelFetchTs = Date.now().valueOf();
      const searchParams = new URLSearchParams({ ...timeParams });
      queries == null ? void 0 : queries.forEach((q) => {
        const visualQuery = buildVisualQueryFromString(q.expr);
        if (visualQuery.query.metric !== "") {
          const isUtf8Metric = !isValidLegacyName(visualQuery.query.metric);
          searchParams.append("match[]", isUtf8Metric ? `{"${visualQuery.query.metric}"}` : visualQuery.query.metric);
          if (visualQuery.query.binaryQueries) {
            visualQuery.query.binaryQueries.forEach((bq) => {
              searchParams.append("match[]", isUtf8Metric ? `{"${bq.query.metric}"}` : bq.query.metric);
            });
          }
        }
      });
      if (this.datasource.httpMethod === "GET") {
        url += `?${searchParams.toString()}`;
      }
      const res = await this.request(url, [], searchParams, this.getDefaultCacheHeaders());
      if (Array.isArray(res)) {
        this.labelKeys = res.slice().sort();
        return [...this.labelKeys];
      }
      return [];
    };
    /**
     * Gets series values
     * Function to replace old getSeries calls in a way that will provide faster endpoints
     * for new prometheus instances, while maintaining backward compatability
     */
    this.getSeriesValues = async (timeRange, labelName, selector) => {
      var _a;
      if (!this.datasource.hasLabelsMatchAPISupport()) {
        const data = await this.getSeries(timeRange, selector);
        return (_a = data[removeQuotesIfExist(labelName)]) != null ? _a : [];
      }
      return await this.fetchSeriesValuesWithMatch(timeRange, labelName, selector);
    };
    /**
     * Fetches all values for a label, with optional match[]
     * @param name
     * @param match
     * @param timeRange
     * @param requestId
     */
    this.fetchSeriesValuesWithMatch = async (timeRange, name, match, requestId) => {
      const interpolatedName = name ? this.datasource.interpolateString(name) : null;
      const interpolatedMatch = match ? this.datasource.interpolateString(match) : null;
      const range = this.datasource.getAdjustedInterval(timeRange);
      const urlParams = {
        ...range,
        ...interpolatedMatch && { "match[]": interpolatedMatch }
      };
      let requestOptions = {
        ...this.getDefaultCacheHeaders(),
        ...requestId && { requestId }
      };
      if (!Object.keys(requestOptions).length) {
        requestOptions = void 0;
      }
      const interpolatedAndEscapedName = escapeForUtf8Support(removeQuotesIfExist(interpolatedName != null ? interpolatedName : ""));
      const value = await this.request(
        `/api/v1/label/${interpolatedAndEscapedName}/values`,
        [],
        urlParams,
        requestOptions
      );
      return value != null ? value : [];
    };
    /**
     * Gets series labels
     * Function to replace old getSeries calls in a way that will provide faster endpoints for new prometheus instances,
     * while maintaining backward compatability. The old API call got the labels and the values in a single query,
     * but with the new query we need two calls, one to get the labels, and another to get the values.
     *
     * @param selector
     * @param otherLabels
     */
    this.getSeriesLabels = async (timeRange, selector, otherLabels) => {
      let possibleLabelNames, data;
      if (!this.datasource.hasLabelsMatchAPISupport()) {
        data = await this.getSeries(timeRange, selector);
        possibleLabelNames = Object.keys(data);
      } else {
        otherLabels.push({ name: "__name__", value: "", op: "!=" });
        data = await this.fetchSeriesLabelsMatch(timeRange, selector);
        possibleLabelNames = Object.keys(data);
      }
      const usedLabelNames = new Set(otherLabels.map((l) => l.name));
      return possibleLabelNames.filter((l) => !usedLabelNames.has(l));
    };
    /**
     * Fetch labels using the best endpoint that datasource supports.
     * This is cached by its args but also by the global timeRange currently selected as they can change over requested time.
     */
    this.fetchLabelsWithMatch = async (timeRange, name, withName) => {
      if (this.datasource.hasLabelsMatchAPISupport()) {
        return this.fetchSeriesLabelsMatch(timeRange, name, withName);
      } else {
        return this.fetchSeriesLabels(timeRange, name, withName, REMOVE_SERIES_LIMIT);
      }
    };
    /**
     * Fetch labels for a series using /series endpoint. This is cached by its args but also by the global timeRange currently selected as
     * they can change over requested time.
     */
    this.fetchSeriesLabels = async (timeRange, name, withName, withLimit) => {
      const interpolatedName = this.datasource.interpolateString(name);
      const range = this.datasource.getAdjustedInterval(timeRange);
      let urlParams = {
        ...range,
        "match[]": interpolatedName
      };
      if (withLimit !== "none") {
        urlParams = { ...urlParams, limit: withLimit != null ? withLimit : DEFAULT_SERIES_LIMIT };
      }
      const url = `/api/v1/series`;
      const data = await this.request(url, [], urlParams, this.getDefaultCacheHeaders());
      const { values } = processLabels(data, withName);
      return values;
    };
    /**
     * Fetch labels for a series using /labels endpoint.  This is cached by its args but also by the global timeRange currently selected as
     * they can change over requested time.
     */
    this.fetchSeriesLabelsMatch = async (timeRange, name, withName) => {
      const interpolatedName = this.datasource.interpolateString(name);
      const range = this.datasource.getAdjustedInterval(timeRange);
      const urlParams = {
        ...range,
        "match[]": interpolatedName
      };
      const url = `/api/v1/labels`;
      const data = await this.request(url, [], urlParams, this.getDefaultCacheHeaders());
      return data.reduce((ac, a) => ({ ...ac, [a]: "" }), {});
    };
    /**
     * Fetch series for a selector. Use this for raw results. Use fetchSeriesLabels() to get labels.
     */
    this.fetchSeries = async (timeRange, match) => {
      const url = "/api/v1/series";
      const range = this.datasource.getTimeRangeParams(timeRange);
      const params = { ...range, "match[]": match };
      return await this.request(url, {}, params, this.getDefaultCacheHeaders());
    };
    /**
     * Fetch this only one as we assume this won't change over time. This is cached differently from fetchSeriesLabels
     * because we can cache more aggressively here and also we do not want to invalidate this cache the same way as in
     * fetchSeriesLabels.
     */
    this.fetchDefaultSeries = once(async (timeRange) => {
      const values = await Promise.all(DEFAULT_KEYS.map((key) => this.fetchLabelValues(timeRange, key)));
      return DEFAULT_KEYS.reduce((acc, key, i) => ({ ...acc, [key]: values[i] }), {});
    });
    /**
     * Fetch labels or values for a label based on the queries, scopes, filters and time range
     * @param timeRange
     * @param queries
     * @param scopes
     * @param adhocFilters
     * @param labelName
     * @param limit
     * @param requestId
     */
    this.fetchSuggestions = async (timeRange, queries, scopes, adhocFilters, labelName, limit, requestId) => {
      var _a;
      if (!timeRange) {
        timeRange = getDefaultTimeRange();
      }
      const url = "/suggestions";
      const timeParams = this.datasource.getAdjustedInterval(timeRange);
      const value = await this.request(
        url,
        [],
        {
          labelName,
          queries: queries == null ? void 0 : queries.map(
            (q) => this.datasource.interpolateString(q.expr, {
              ...this.datasource.getIntervalVars(),
              ...this.datasource.getRangeScopedVars(timeRange)
            })
          ),
          scopes: scopes == null ? void 0 : scopes.reduce((acc, scope) => {
            acc.push(...scope.spec.filters);
            return acc;
          }, []),
          adhocFilters: adhocFilters == null ? void 0 : adhocFilters.map((filter) => ({
            key: filter.key,
            operator: scopeFilterOperatorMap[filter.operator],
            value: filter.value,
            values: filter.values
          })),
          limit,
          ...timeParams
        },
        {
          ...requestId && { requestId },
          headers: {
            ...(_a = this.getDefaultCacheHeaders()) == null ? void 0 : _a.headers,
            "Content-Type": "application/json"
          },
          method: "POST"
        }
      );
      return value != null ? value : [];
    };
    this.datasource = datasource;
    this.histogramMetrics = [];
    this.metrics = [];
    Object.assign(this, initialValues);
  }
  getDefaultCacheHeaders() {
    if (this.datasource.cacheLevel !== PrometheusCacheLevel.None) {
      return buildCacheHeaders(this.datasource.getCacheDurationInMinutes() * 60);
    }
    return;
  }
  // Strip syntax chars so that typeahead suggestions can work on clean inputs
  cleanText(s) {
    const parts = s.split(PREFIX_DELIMITER_REGEX);
    const last = parts.pop();
    return last.trimStart().replace(/"$/, "").replace(/^"/, "");
  }
  get syntax() {
    return promqlGrammar;
  }
  async loadMetricsMetadata() {
    const headers = buildCacheHeaders(this.datasource.getDaysToCacheMetadata() * secondsInDay);
    this.metricsMetadata = fixSummariesMetadata(
      await this.request(
        "/api/v1/metadata",
        {},
        {},
        {
          showErrorAlert: false,
          ...headers
        }
      )
    );
  }
  getLabelKeys() {
    return this.labelKeys;
  }
  importFromAbstractQuery(labelBasedQuery) {
    return toPromLikeQuery(labelBasedQuery);
  }
  exportToAbstractQuery(query) {
    const promQuery = query.expr;
    if (!promQuery || promQuery.length === 0) {
      return { refId: query.refId, labelMatchers: [] };
    }
    const tokens = Prism.tokenize(promQuery, promqlGrammar);
    const labelMatchers = extractLabelMatchers(tokens);
    const nameLabelValue = getNameLabelValue(promQuery, tokens);
    if (nameLabelValue && nameLabelValue.length > 0) {
      labelMatchers.push({
        name: "__name__",
        operator: AbstractLabelOperator.Equal,
        value: nameLabelValue
      });
    }
    return {
      refId: query.refId,
      labelMatchers
    };
  }
  async getSeries(timeRange, selector, withName) {
    if (this.datasource.lookupsDisabled) {
      return {};
    }
    try {
      if (selector === EMPTY_SELECTOR) {
        return await this.fetchDefaultSeries(timeRange);
      } else {
        return await this.fetchSeriesLabels(timeRange, selector, withName, REMOVE_SERIES_LIMIT);
      }
    } catch (error) {
      console.error(error);
      return {};
    }
  }
  async getLabelValues(range, key) {
    return await this.fetchLabelValues(range, key);
  }
}
function getNameLabelValue(promQuery, tokens) {
  let nameLabelValue = "";
  for (const token of tokens) {
    if (typeof token === "string") {
      nameLabelValue = token;
      break;
    }
  }
  return nameLabelValue;
}
function isCancelledError(error) {
  return typeof error === "object" && error !== null && "cancelled" in error && error.cancelled === true;
}
function removeQuotesIfExist(input) {
  var _a;
  const match = input.match(/^"(.*)"$/);
  return (_a = match == null ? void 0 : match[1]) != null ? _a : input;
}

export { SUGGESTIONS_LIMIT, PromQlLanguageProvider as default, getMetadataHelp, getMetadataString, getMetadataType, removeQuotesIfExist };
//# sourceMappingURL=language_provider.mjs.map
