import { map } from 'rxjs/operators';
import { FieldType } from '../../types/dataFrame.mjs';
import { TransformationApplicabilityLevels } from '../../types/transformations.mjs';
import { fieldMatchers } from '../matchers.mjs';
import { FieldMatcherID } from '../matchers/ids.mjs';
import { DataTransformerID } from './ids.mjs';

"use strict";
var FormatStringOutput = /* @__PURE__ */ ((FormatStringOutput2) => {
  FormatStringOutput2["UpperCase"] = "Upper Case";
  FormatStringOutput2["LowerCase"] = "Lower Case";
  FormatStringOutput2["SentenceCase"] = "Sentence Case";
  FormatStringOutput2["TitleCase"] = "Title Case";
  FormatStringOutput2["PascalCase"] = "Pascal Case";
  FormatStringOutput2["CamelCase"] = "Camel Case";
  FormatStringOutput2["SnakeCase"] = "Snake Case";
  FormatStringOutput2["KebabCase"] = "Kebab Case";
  FormatStringOutput2["Trim"] = "Trim";
  FormatStringOutput2["Substring"] = "Substring";
  return FormatStringOutput2;
})(FormatStringOutput || {});
const splitToCapitalWords = (input) => {
  const arr = input.split(" ");
  for (let i = 0; i < arr.length; i++) {
    arr[i] = arr[i].charAt(0).toUpperCase() + arr[i].slice(1).toLowerCase();
  }
  return arr;
};
const getFormatStringFunction = (options) => {
  return (field) => field.values.map((value) => {
    switch (options.outputFormat) {
      case "Upper Case" /* UpperCase */:
        return value.toUpperCase();
      case "Lower Case" /* LowerCase */:
        return value.toLowerCase();
      case "Sentence Case" /* SentenceCase */:
        return value.charAt(0).toUpperCase() + value.slice(1);
      case "Title Case" /* TitleCase */:
        return splitToCapitalWords(value).join(" ");
      case "Pascal Case" /* PascalCase */:
        return splitToCapitalWords(value).join("");
      case "Camel Case" /* CamelCase */:
        value = splitToCapitalWords(value).join("");
        return value.charAt(0).toLowerCase() + value.slice(1);
      case "Snake Case" /* SnakeCase */:
        return value.toLowerCase().split(" ").join("_");
      case "Kebab Case" /* KebabCase */:
        return value.toLowerCase().split(" ").join("-");
      case "Trim" /* Trim */:
        return value.trim();
      case "Substring" /* Substring */:
        return value.substring(options.substringStart, options.substringEnd);
    }
  });
};
const formatStringTransformer = {
  id: DataTransformerID.formatString,
  name: "Format string",
  description: "Manipulate string fields formatting",
  defaultOptions: { stringField: "", outputFormat: "Upper Case" /* UpperCase */ },
  isApplicable: (data) => {
    for (const frame of data) {
      for (const field of frame.fields) {
        if (field.type === "string") {
          return TransformationApplicabilityLevels.Applicable;
        }
      }
    }
    return TransformationApplicabilityLevels.NotApplicable;
  },
  operator: (options) => (source) => source.pipe(
    map((data) => {
      if (data.length === 0) {
        return data;
      }
      const fieldMatches = fieldMatchers.get(FieldMatcherID.byName).get(options.stringField);
      const formatStringFunction = getFormatStringFunction(options);
      const formatter = createStringFormatter(fieldMatches, formatStringFunction);
      return data.map((frame) => ({
        ...frame,
        fields: formatter(frame, data)
      }));
    })
  )
};
const createStringFormatter = (fieldMatches, formatStringFunction) => (frame, allFrames) => {
  return frame.fields.map((field) => {
    if (fieldMatches(field, frame, allFrames)) {
      const newVals = formatStringFunction(field);
      return {
        ...field,
        type: FieldType.string,
        values: newVals
      };
    }
    return field;
  });
};

export { FormatStringOutput, createStringFormatter, formatStringTransformer, getFormatStringFunction };
//# sourceMappingURL=formatString.mjs.map
