import { jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { PureComponent } from 'react';
import { monacoLanguageRegistry } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import '../../themes/index.mjs';
import { ReactMonacoEditorLazy } from './ReactMonacoEditorLazy.mjs';
import { registerSuggestions } from './suggestions.mjs';
import { withTheme2 } from '../../themes/ThemeContext.mjs';

"use strict";
class UnthemedCodeEditor extends PureComponent {
  constructor(props) {
    super(props);
    this.loadCustomLanguage = () => {
      const { language } = this.props;
      const customLanguage = monacoLanguageRegistry.getIfExists(language);
      if (customLanguage) {
        return customLanguage.init();
      }
      return Promise.resolve();
    };
    // This is replaced with a real function when the actual editor mounts
    this.getEditorValue = () => "";
    this.onBlur = () => {
      const { onBlur } = this.props;
      if (onBlur) {
        onBlur(this.getEditorValue());
      }
    };
    this.onFocus = () => {
      const { onFocus } = this.props;
      if (onFocus) {
        onFocus(this.getEditorValue());
      }
    };
    this.onSave = () => {
      const { onSave } = this.props;
      if (onSave) {
        onSave(this.getEditorValue());
      }
    };
    this.handleBeforeMount = (monaco) => {
      this.monaco = monaco;
      const { onBeforeEditorMount } = this.props;
      onBeforeEditorMount == null ? void 0 : onBeforeEditorMount(monaco);
    };
    this.handleOnMount = (editor, monaco) => {
      var _a, _b;
      const { getSuggestions, language, onChange, onEditorDidMount } = this.props;
      this.modelId = (_a = editor.getModel()) == null ? void 0 : _a.id;
      this.getEditorValue = () => editor.getValue();
      if (getSuggestions && this.modelId) {
        this.completionCancel = registerSuggestions(monaco, language, getSuggestions, this.modelId);
      }
      editor.onKeyDown((e) => {
        if (e.keyCode === monaco.KeyCode.KeyS && (e.ctrlKey || e.metaKey)) {
          e.preventDefault();
          this.onSave();
        }
      });
      if (onChange) {
        (_b = editor.getModel()) == null ? void 0 : _b.onDidChangeContent(() => onChange(editor.getValue()));
      }
      if (onEditorDidMount) {
        onEditorDidMount(editor, monaco);
      }
    };
  }
  componentWillUnmount() {
    var _a, _b;
    if (this.completionCancel) {
      this.completionCancel.dispose();
    }
    (_b = (_a = this.props).onEditorWillUnmount) == null ? void 0 : _b.call(_a);
  }
  componentDidUpdate(oldProps) {
    const { getSuggestions, language } = this.props;
    const newLanguage = oldProps.language !== language;
    const newGetSuggestions = oldProps.getSuggestions !== getSuggestions;
    if (newGetSuggestions || newLanguage) {
      if (this.completionCancel) {
        this.completionCancel.dispose();
      }
      if (!this.monaco) {
        console.warn("Monaco instance not loaded yet");
        return;
      }
      if (getSuggestions && this.modelId) {
        this.completionCancel = registerSuggestions(this.monaco, language, getSuggestions, this.modelId);
      }
    }
    if (newLanguage) {
      this.loadCustomLanguage();
    }
  }
  render() {
    var _a, _b;
    const { theme, language, width, height, showMiniMap, showLineNumbers, readOnly, wordWrap, monacoOptions } = this.props;
    const { alwaysConsumeMouseWheel, ...restMonacoOptions } = monacoOptions != null ? monacoOptions : {};
    const value = (_a = this.props.value) != null ? _a : "";
    const longText = value.length > 100;
    const containerStyles = (_b = this.props.containerStyles) != null ? _b : getStyles(theme).container;
    const options = {
      wordWrap: wordWrap ? "on" : "off",
      tabSize: 2,
      codeLens: false,
      contextmenu: false,
      minimap: {
        enabled: longText && showMiniMap,
        renderCharacters: false
      },
      readOnly,
      lineNumbersMinChars: 4,
      lineDecorationsWidth: 1 * theme.spacing.gridSize,
      overviewRulerBorder: false,
      automaticLayout: true,
      padding: {
        top: 0.5 * theme.spacing.gridSize,
        bottom: 0.5 * theme.spacing.gridSize
      },
      fixedOverflowWidgets: true,
      // Ensures suggestions menu is drawn on top
      scrollbar: {
        alwaysConsumeMouseWheel: alwaysConsumeMouseWheel != null ? alwaysConsumeMouseWheel : false
      }
    };
    if (!showLineNumbers) {
      options.glyphMargin = false;
      options.folding = false;
      options.lineNumbers = "off";
      options.lineNumbersMinChars = 0;
    }
    return /* @__PURE__ */ jsx(
      "div",
      {
        className: containerStyles,
        onFocus: this.onFocus,
        onBlur: this.onBlur,
        "data-testid": selectors.components.CodeEditor.container,
        children: /* @__PURE__ */ jsx(
          ReactMonacoEditorLazy,
          {
            width,
            height,
            language,
            value,
            options: {
              ...options,
              ...restMonacoOptions != null ? restMonacoOptions : {}
            },
            beforeMount: this.handleBeforeMount,
            onMount: this.handleOnMount,
            keepCurrentModel: true
          }
        )
      }
    );
  }
}
const CodeEditor = withTheme2(UnthemedCodeEditor);
const getStyles = (theme) => {
  return {
    container: css({
      borderRadius: theme.shape.radius.default,
      border: `1px solid ${theme.components.input.borderColor}`
    })
  };
};

export { CodeEditor };
//# sourceMappingURL=CodeEditor.mjs.map
