import { defaults } from 'lodash';
import { map } from 'rxjs/operators';
import { getTimeField } from '../../dataframe/processDataFrame.mjs';
import { getFieldDisplayName } from '../../field/fieldState.mjs';
import { NullValueMode } from '../../types/data.mjs';
import { FieldType } from '../../types/dataFrame.mjs';
import { BinaryOperationID, binaryOperators } from '../../utils/binaryOperators.mjs';
import { UnaryOperationID, unaryOperators } from '../../utils/unaryOperators.mjs';
import { ReducerID, fieldReducers, doStandardCalcs } from '../fieldReducer.mjs';
import { getFieldMatcher } from '../matchers.mjs';
import { FieldMatcherID } from '../matchers/ids.mjs';
import { ensureColumnsTransformer } from './ensureColumns.mjs';
import { DataTransformerID } from './ids.mjs';
import { noopTransformer } from './noop.mjs';

"use strict";
var CalculateFieldMode = /* @__PURE__ */ ((CalculateFieldMode2) => {
  CalculateFieldMode2["ReduceRow"] = "reduceRow";
  CalculateFieldMode2["CumulativeFunctions"] = "cumulativeFunctions";
  CalculateFieldMode2["WindowFunctions"] = "windowFunctions";
  CalculateFieldMode2["BinaryOperation"] = "binary";
  CalculateFieldMode2["UnaryOperation"] = "unary";
  CalculateFieldMode2["Index"] = "index";
  return CalculateFieldMode2;
})(CalculateFieldMode || {});
var WindowSizeMode = /* @__PURE__ */ ((WindowSizeMode2) => {
  WindowSizeMode2["Percentage"] = "percentage";
  WindowSizeMode2["Fixed"] = "fixed";
  return WindowSizeMode2;
})(WindowSizeMode || {});
var WindowAlignment = /* @__PURE__ */ ((WindowAlignment2) => {
  WindowAlignment2["Trailing"] = "trailing";
  WindowAlignment2["Centered"] = "centered";
  return WindowAlignment2;
})(WindowAlignment || {});
const defaultReduceOptions = {
  reducer: ReducerID.sum
};
const defaultWindowOptions = {
  reducer: ReducerID.mean,
  windowAlignment: "trailing" /* Trailing */,
  windowSizeMode: "percentage" /* Percentage */,
  windowSize: 0.1
};
const defaultBinaryOptions = {
  left: { fixed: "" },
  operator: BinaryOperationID.Add,
  right: { fixed: "" }
};
const defaultUnaryOptions = {
  operator: UnaryOperationID.Abs,
  fieldName: ""
};
const calculateFieldTransformer = {
  id: DataTransformerID.calculateField,
  name: "Add field from calculation",
  description: "Use the row values to calculate a new field",
  defaultOptions: {
    mode: "reduceRow" /* ReduceRow */,
    reduce: {
      reducer: ReducerID.sum
    }
  },
  operator: (options, ctx) => (outerSource) => {
    var _a, _b;
    const mode = (_a = options.mode) != null ? _a : "reduceRow" /* ReduceRow */;
    const asTimeSeries = options.timeSeries !== false;
    const right = (_b = options.binary) == null ? void 0 : _b.right;
    const rightVal = typeof right === "string" ? right : typeof right === "object" ? right.fixed : void 0;
    const isBinaryFixed = mode === "binary" /* BinaryOperation */ && !Number.isNaN(Number(rightVal));
    const needsSingleFrame = asTimeSeries && !isBinaryFixed;
    const operator = needsSingleFrame ? ensureColumnsTransformer.operator(null, ctx) : noopTransformer.operator({}, ctx);
    return outerSource.pipe(
      operator,
      map((data) => {
        var _a2, _b2, _c, _d, _e, _f, _g, _h, _i;
        let creator = void 0;
        switch (mode) {
          case "reduceRow" /* ReduceRow */:
            creator = getReduceRowCreator(defaults(options.reduce, defaultReduceOptions), data);
            break;
          case "cumulativeFunctions" /* CumulativeFunctions */:
            creator = getCumulativeCreator(defaults(options.cumulative, defaultReduceOptions), data);
            break;
          case "windowFunctions" /* WindowFunctions */:
            creator = getWindowCreator(defaults(options.window, defaultWindowOptions), data);
            break;
          case "unary" /* UnaryOperation */:
            creator = getUnaryCreator(defaults(options.unary, defaultUnaryOptions), data);
            break;
          case "binary" /* BinaryOperation */:
            const fieldNames = [];
            data.map((frame) => {
              frame.fields.map((field) => {
                fieldNames.push(field.name);
              });
            });
            const binaryOptions = {
              left: checkBinaryValueType((_b2 = (_a2 = options.binary) == null ? void 0 : _a2.left) != null ? _b2 : "", fieldNames),
              operator: (_d = (_c = options.binary) == null ? void 0 : _c.operator) != null ? _d : defaultBinaryOptions.operator,
              right: checkBinaryValueType((_f = (_e = options.binary) == null ? void 0 : _e.right) != null ? _f : "", fieldNames)
            };
            options.binary = binaryOptions;
            if (((_h = (_g = binaryOptions.left) == null ? void 0 : _g.matcher) == null ? void 0 : _h.id) && ((_i = binaryOptions.left) == null ? void 0 : _i.matcher.id) === FieldMatcherID.byType) {
              const fieldType = binaryOptions.left.matcher.options;
              const operator2 = binaryOperators.getIfExists(binaryOptions.operator);
              const outFrames2 = data.map((frame) => {
                const { timeField } = getTimeField(frame);
                const newFields = [];
                let didAddNewFields = false;
                if (timeField && options.timeSeries !== false) {
                  newFields.push(timeField);
                }
                frame.fields.map((field, index) => {
                  var _a3, _b3, _c2, _d2, _e2, _f2, _g2;
                  if (!options.replaceFields && !newFields.includes(field)) {
                    newFields.push(field);
                  }
                  if (field.type === fieldType) {
                    const left = field.values;
                    const right2 = findFieldValuesWithNameOrConstant(
                      frame,
                      (_a3 = binaryOptions.right) != null ? _a3 : defaultBinaryOptions.right,
                      data,
                      ctx
                    );
                    if (!left || !right2 || !operator2) {
                      return void 0;
                    }
                    const arr = new Array(left.length);
                    for (let i = 0; i < arr.length; i++) {
                      arr[i] = operator2.operation(left[i], right2[i]);
                    }
                    const newField = {
                      ...field,
                      name: `${field.name} ${(_c2 = (_b3 = options.binary) == null ? void 0 : _b3.operator) != null ? _c2 : ""} ${(_g2 = (_e2 = (_d2 = options.binary) == null ? void 0 : _d2.right.matcher) == null ? void 0 : _e2.options) != null ? _g2 : (_f2 = options.binary) == null ? void 0 : _f2.right.fixed}`,
                      values: arr
                    };
                    delete newField.state;
                    newFields.push(newField);
                    didAddNewFields = true;
                  }
                });
                if (options.replaceFields && !didAddNewFields) {
                  return void 0;
                }
                return { ...frame, fields: newFields };
              });
              return outFrames2.filter((frame) => frame != null);
            } else {
              creator = getBinaryCreator(defaults(binaryOptions, defaultBinaryOptions), data, ctx);
            }
            break;
          case "index" /* Index */:
            return data.map((frame) => {
              var _a3, _b3, _c2;
              const indexArr = [...Array(frame.length).keys()];
              if ((_a3 = options.index) == null ? void 0 : _a3.asPercentile) {
                for (let i = 0; i < indexArr.length; i++) {
                  indexArr[i] = indexArr[i] / indexArr.length;
                }
              }
              const f = {
                name: (_b3 = options.alias) != null ? _b3 : "Row",
                type: FieldType.number,
                values: indexArr,
                config: ((_c2 = options.index) == null ? void 0 : _c2.asPercentile) ? { unit: "percentunit" } : {}
              };
              return {
                ...frame,
                fields: options.replaceFields ? [f] : [...frame.fields, f]
              };
            });
        }
        if (!creator) {
          return data;
        }
        const outFrames = data.map((frame) => {
          const values = creator(frame);
          if (!values) {
            if (options.replaceFields) {
              return void 0;
            }
            return frame;
          }
          const field = {
            name: getNameFromOptions(options),
            type: FieldType.number,
            config: {},
            values
          };
          let fields = [];
          if (options.replaceFields) {
            const { timeField } = getTimeField(frame);
            if (timeField && options.timeSeries !== false) {
              fields = [timeField, field];
            } else {
              fields = [field];
            }
          } else {
            fields = [...frame.fields, field];
          }
          return {
            ...frame,
            fields
          };
        });
        return outFrames.filter((frame) => frame != null);
      })
    );
  }
};
function getWindowCreator(options, allFrames) {
  if (options.windowSize <= 0) {
    throw new Error("Add field from calculation transformation - Window size must be larger than 0");
  }
  let matcher = getFieldMatcher({
    id: FieldMatcherID.numeric
  });
  if (options.field) {
    matcher = getFieldMatcher({
      id: FieldMatcherID.byNames,
      options: {
        names: [options.field]
      }
    });
  }
  return (frame) => {
    const window = Math.ceil(
      options.windowSize * (options.windowSizeMode === "percentage" /* Percentage */ ? frame.length : 1)
    );
    let selectedField = null;
    for (const field of frame.fields) {
      if (matcher(field, frame, allFrames)) {
        selectedField = field;
        break;
      }
    }
    if (!selectedField) {
      return;
    }
    if (![ReducerID.mean, ReducerID.stdDev, ReducerID.variance].includes(options.reducer)) {
      throw new Error(`Add field from calculation transformation - Unsupported reducer: ${options.reducer}`);
    }
    if (options.windowAlignment === "centered" /* Centered */) {
      return getCenteredWindowValues(frame, options.reducer, selectedField, window);
    } else {
      return getTrailingWindowValues(frame, options.reducer, selectedField, window);
    }
  };
}
function getTrailingWindowValues(frame, reducer, selectedField, window) {
  const vals = [];
  let sum = 0;
  let count = 0;
  for (let i = 0; i < frame.length; i++) {
    if (reducer === ReducerID.mean) {
      const currentValue = selectedField.values[i];
      if (currentValue !== null && currentValue !== void 0) {
        count++;
        sum += currentValue;
        if (i > window - 1) {
          const value = selectedField.values[i - window];
          if (value != null) {
            sum -= value;
            count--;
          }
        }
      }
      vals.push(count === 0 ? 0 : sum / count);
    } else if (reducer === ReducerID.variance) {
      const start = Math.max(0, i - window + 1);
      const end = i + 1;
      vals.push(calculateVariance(selectedField.values.slice(start, end)));
    } else if (reducer === ReducerID.stdDev) {
      const start = Math.max(0, i - window + 1);
      const end = i + 1;
      vals.push(calculateStdDev(selectedField.values.slice(start, end)));
    }
  }
  return vals;
}
function getCenteredWindowValues(frame, reducer, selectedField, window) {
  const vals = [];
  let sum = 0;
  let count = 0;
  const leadingPartOfWindow = Math.ceil(window / 2) - 1;
  const trailingPartOfWindow = Math.floor(window / 2);
  for (let i = 0; i < frame.length; i++) {
    const first = i - trailingPartOfWindow;
    const last = i + leadingPartOfWindow;
    if (reducer === ReducerID.mean) {
      if (i === 0) {
        for (let x = 0; x < leadingPartOfWindow + 1 && x < selectedField.values.length; x++) {
          if (selectedField.values[x] != null) {
            sum += selectedField.values[x];
            count++;
          }
        }
      } else {
        if (last < selectedField.values.length) {
          if (selectedField.values[last] != null) {
            sum += selectedField.values[last];
            count++;
          }
        }
        if (first > 0) {
          if (selectedField.values[first - 1] != null) {
            sum -= selectedField.values[first - 1];
            count--;
          }
        }
      }
      vals.push(count === 0 ? 0 : sum / count);
    } else if (reducer === ReducerID.variance) {
      const windowVals = selectedField.values.slice(
        Math.max(0, first),
        Math.min(last + 1, selectedField.values.length)
      );
      vals.push(calculateVariance(windowVals));
    } else if (reducer === ReducerID.stdDev) {
      const windowVals = selectedField.values.slice(
        Math.max(0, first),
        Math.min(last + 1, selectedField.values.length)
      );
      vals.push(calculateStdDev(windowVals));
    }
  }
  return vals;
}
function calculateVariance(vals) {
  if (vals.length < 1) {
    return 0;
  }
  let squareSum = 0;
  let runningMean = 0;
  let nonNullCount = 0;
  for (let i = 0; i < vals.length; i++) {
    const currentValue = vals[i];
    if (currentValue != null) {
      nonNullCount++;
      let _oldMean = runningMean;
      runningMean += (currentValue - _oldMean) / nonNullCount;
      squareSum += (currentValue - _oldMean) * (currentValue - runningMean);
    }
  }
  if (nonNullCount === 0) {
    return 0;
  }
  const variance = squareSum / nonNullCount;
  return variance;
}
function calculateStdDev(vals) {
  return Math.sqrt(calculateVariance(vals));
}
function getCumulativeCreator(options, allFrames) {
  let matcher = getFieldMatcher({
    id: FieldMatcherID.numeric
  });
  if (options.field) {
    matcher = getFieldMatcher({
      id: FieldMatcherID.byNames,
      options: {
        names: [options.field]
      }
    });
  }
  if (![ReducerID.mean, ReducerID.sum].includes(options.reducer)) {
    throw new Error(`Add field from calculation transformation - Unsupported reducer: ${options.reducer}`);
  }
  return (frame) => {
    var _a;
    let selectedField = null;
    for (const field of frame.fields) {
      if (matcher(field, frame, allFrames)) {
        selectedField = field;
        break;
      }
    }
    if (!selectedField) {
      return;
    }
    const vals = [];
    let total = 0;
    for (let i = 0; i < frame.length; i++) {
      total += (_a = selectedField.values[i]) != null ? _a : 0;
      if (options.reducer === ReducerID.sum) {
        vals.push(total);
      } else if (options.reducer === ReducerID.mean) {
        vals.push(total / (i + 1));
      }
    }
    return vals;
  };
}
function getReduceRowCreator(options, allFrames) {
  var _a;
  let matcher = getFieldMatcher({
    id: FieldMatcherID.numeric
  });
  if (options.include && options.include.length) {
    matcher = getFieldMatcher({
      id: FieldMatcherID.byNames,
      options: {
        names: options.include
      }
    });
  }
  const info = fieldReducers.get(options.reducer);
  if (!info) {
    throw new Error(`Unknown reducer: ${options.reducer}`);
  }
  const reducer = (_a = info.reduce) != null ? _a : doStandardCalcs;
  const ignoreNulls = options.nullValueMode === NullValueMode.Ignore;
  const nullAsZero = options.nullValueMode === NullValueMode.AsZero;
  return (frame) => {
    const columns = [];
    for (const field of frame.fields) {
      if (matcher(field, frame, allFrames)) {
        columns.push(field.values);
      }
    }
    const size = columns.length;
    const row = {
      name: "temp",
      values: new Array(size),
      type: FieldType.number,
      config: {}
    };
    const vals = [];
    for (let i = 0; i < frame.length; i++) {
      for (let j = 0; j < size; j++) {
        row.values[j] = columns[j][i];
      }
      vals.push(reducer(row, ignoreNulls, nullAsZero)[options.reducer]);
    }
    return vals;
  };
}
function findFieldValuesWithNameOrConstant(frame, value, allFrames, ctx) {
  var _a, _b, _c, _d;
  if (!value) {
    return void 0;
  }
  if (value.matcher && value.matcher.id === FieldMatcherID.byName) {
    const name = (_a = value.matcher.options) != null ? _a : "";
    for (const f of frame.fields) {
      if (name === getFieldDisplayName(f, frame, allFrames)) {
        if (f.type === FieldType.boolean) {
          return f.values.map((v2) => v2 ? 1 : 0);
        }
        return f.values;
      }
    }
  }
  const v = parseFloat((_d = (_c = value.fixed) != null ? _c : (_b = value.matcher) == null ? void 0 : _b.options) != null ? _d : "");
  if (!isNaN(v)) {
    return new Array(frame.length).fill(v);
  }
  return void 0;
}
function getBinaryCreator(options, allFrames, ctx) {
  const operator = binaryOperators.getIfExists(options.operator);
  return (frame) => {
    const left = findFieldValuesWithNameOrConstant(frame, options.left, allFrames, ctx);
    const right = findFieldValuesWithNameOrConstant(frame, options.right, allFrames, ctx);
    if (!left || !right || !operator) {
      return void 0;
    }
    const arr = new Array(left.length);
    for (let i = 0; i < arr.length; i++) {
      arr[i] = operator.operation(left[i], right[i]);
    }
    return arr;
  };
}
function checkBinaryValueType(value, names) {
  if (typeof value === "string") {
    if (isNaN(Number(value))) {
      return { matcher: { id: FieldMatcherID.byName, options: value } };
    } else {
      if (names.includes(value)) {
        return { matcher: { id: FieldMatcherID.byName, options: value } };
      } else {
        return { fixed: value };
      }
    }
  }
  return value;
}
function getUnaryCreator(options, allFrames) {
  const operator = unaryOperators.getIfExists(options.operator);
  return (frame) => {
    let value = [];
    for (const f of frame.fields) {
      if (options.fieldName === getFieldDisplayName(f, frame, allFrames) && f.type === FieldType.number) {
        value = f.values;
      }
    }
    if (!value.length || !operator) {
      return void 0;
    }
    const arr = new Array(value.length);
    for (let i = 0; i < arr.length; i++) {
      arr[i] = operator.operation(value[i]);
    }
    return arr;
  };
}
function getNameFromOptions(options) {
  var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j, _k, _l, _m, _n, _o, _p, _q;
  if ((_a = options.alias) == null ? void 0 : _a.length) {
    return options.alias;
  }
  switch (options.mode) {
    case "cumulativeFunctions" /* CumulativeFunctions */: {
      const { cumulative } = options;
      return `cumulative ${(_b = cumulative == null ? void 0 : cumulative.reducer) != null ? _b : ""}${(cumulative == null ? void 0 : cumulative.field) ? `(${cumulative.field})` : ""}`;
    }
    case "windowFunctions" /* WindowFunctions */: {
      const { window } = options;
      return `${(_c = window == null ? void 0 : window.windowAlignment) != null ? _c : ""} moving ${(_d = window == null ? void 0 : window.reducer) != null ? _d : ""}${(window == null ? void 0 : window.field) ? `(${window.field})` : ""}`;
    }
    case "unary" /* UnaryOperation */: {
      const { unary } = options;
      return `${(_e = unary == null ? void 0 : unary.operator) != null ? _e : ""}${(unary == null ? void 0 : unary.fieldName) ? `(${unary.fieldName})` : ""}`;
    }
    case "binary" /* BinaryOperation */: {
      const { binary } = options;
      const alias = `${(_j = (_i = (_g = (_f = binary == null ? void 0 : binary.left) == null ? void 0 : _f.matcher) == null ? void 0 : _g.options) != null ? _i : (_h = binary == null ? void 0 : binary.left) == null ? void 0 : _h.fixed) != null ? _j : ""} ${(_k = binary == null ? void 0 : binary.operator) != null ? _k : ""} ${(_p = (_o = (_m = (_l = binary == null ? void 0 : binary.right) == null ? void 0 : _l.matcher) == null ? void 0 : _m.options) != null ? _o : (_n = binary == null ? void 0 : binary.right) == null ? void 0 : _n.fixed) != null ? _p : ""}`;
      return alias.replace(/\$/g, "");
    }
    case "reduceRow" /* ReduceRow */:
      {
        const r = fieldReducers.getIfExists((_q = options.reduce) == null ? void 0 : _q.reducer);
        if (r) {
          return r.name;
        }
      }
      break;
    case "index" /* Index */:
      return "Row";
  }
  return "math";
}

export { CalculateFieldMode, WindowAlignment, WindowSizeMode, calculateFieldTransformer, checkBinaryValueType, defaultWindowOptions, getNameFromOptions };
//# sourceMappingURL=calculateField.mjs.map
