import { debounce } from 'lodash';
import { useMemo, useState, useCallback } from 'react';
import { t } from '../../utils/i18n.mjs';
import { itemToString, fuzzyFind } from './filter.mjs';
import { useLatestAsyncCall, StaleResultError } from './useLatestAsyncCall.mjs';

const asyncNoop = () => Promise.resolve([]);
function useOptions(rawOptions, createCustomValue) {
  const isAsync = typeof rawOptions === "function";
  const loadOptions = useLatestAsyncCall(isAsync ? rawOptions : asyncNoop);
  const debouncedLoadOptions = useMemo(
    () => debounce((searchTerm) => {
      return loadOptions(searchTerm).then((options) => {
        setAsyncOptions(options);
        setAsyncLoading(false);
        setAsyncError(false);
      }).catch((error) => {
        if (!(error instanceof StaleResultError)) {
          setAsyncError(true);
          setAsyncLoading(false);
          if (error) {
            console.error("Error loading async options for Combobox", error);
          }
        }
      });
    }, 200),
    [loadOptions]
  );
  const [asyncOptions, setAsyncOptions] = useState([]);
  const [asyncLoading, setAsyncLoading] = useState(false);
  const [asyncError, setAsyncError] = useState(false);
  const [userTypedSearch, setUserTypedSearch] = useState("");
  const addCustomValue = useCallback(
    (opts) => {
      let currentOptions = opts;
      if (createCustomValue && userTypedSearch) {
        const customValueExists = opts.some((opt) => opt.value === userTypedSearch);
        if (!customValueExists) {
          currentOptions = [
            {
              label: userTypedSearch,
              value: userTypedSearch,
              description: t("combobox.custom-value.description", "Use custom value")
            },
            ...currentOptions
          ];
        }
      }
      return currentOptions;
    },
    [createCustomValue, userTypedSearch]
  );
  const updateOptions = useCallback(
    (inputValue) => {
      setUserTypedSearch(inputValue);
      if (isAsync) {
        setAsyncLoading(true);
        debouncedLoadOptions(inputValue);
      }
    },
    [debouncedLoadOptions, isAsync]
  );
  const organizeOptionsByGroup = useCallback((options) => {
    var _a;
    const groupedOptions = /* @__PURE__ */ new Map();
    for (const option of options) {
      const groupExists = groupedOptions.has(option.group);
      if (groupExists) {
        (_a = groupedOptions.get(option.group)) == null ? undefined : _a.push(option);
      } else {
        groupedOptions.set(option.group, [option]);
      }
    }
    const reorganizeOptions = [];
    for (const [group, groupOptions] of groupedOptions) {
      if (!group) {
        continue;
      }
      reorganizeOptions.push(...groupOptions);
    }
    const undefinedGroupOptions = groupedOptions.get(undefined);
    if (undefinedGroupOptions) {
      reorganizeOptions.push(...undefinedGroupOptions);
    }
    return reorganizeOptions;
  }, []);
  const stringifiedOptions = useMemo(() => {
    return isAsync ? [] : rawOptions.map(itemToString);
  }, [isAsync, rawOptions]);
  const finalOptions = useMemo(() => {
    const currentOptions = isAsync ? asyncOptions : fuzzyFind(rawOptions, stringifiedOptions, userTypedSearch);
    const currentOptionsOrganised = organizeOptionsByGroup(currentOptions);
    return addCustomValue(currentOptionsOrganised);
  }, [isAsync, organizeOptionsByGroup, addCustomValue, asyncOptions, rawOptions, userTypedSearch, stringifiedOptions]);
  return { options: finalOptions, updateOptions, asyncLoading, asyncError };
}

export { useOptions };
//# sourceMappingURL=useOptions.mjs.map
