import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { useState, useCallback } from 'react';
import { EditorFieldGroup, EditorField, InputGroup } from '@grafana/plugin-ui';
import { InlineFieldRow, InlineField, Combobox, Button } from '@grafana/ui';
import { regexifyLabelValuesQueryString } from '../parsingUtils.mjs';
import { MetricsModal } from './metrics-modal/MetricsModal.mjs';
import { tracking } from './metrics-modal/state/helpers.mjs';

function MetricCombobox({
  datasource,
  query,
  onChange,
  onGetMetrics,
  labelsFilters,
  variableEditor
}) {
  const [metricsModalOpen, setMetricsModalOpen] = useState(false);
  const getMetricLabels = useCallback(
    async (query2) => {
      const results = await datasource.metricFindQuery(formatKeyValueStringsForLabelValuesQuery(query2, labelsFilters));
      const resultsOptions = results.map((result) => {
        return {
          label: result.text,
          value: result.text
        };
      });
      return resultsOptions;
    },
    [datasource, labelsFilters]
  );
  const onComboboxChange = useCallback(
    (opt) => {
      var _a;
      onChange({ ...query, metric: (_a = opt == null ? undefined : opt.value) != null ? _a : "" });
    },
    [onChange, query]
  );
  const loadOptions = useCallback(
    async (input) => {
      const metrics = input.length ? await getMetricLabels(input) : await onGetMetrics();
      return metrics.map((option) => {
        var _a;
        return {
          label: (_a = option.label) != null ? _a : option.value,
          value: option.value
        };
      });
    },
    [getMetricLabels, onGetMetrics]
  );
  const loadMetricsExplorerMetrics = useCallback(async () => {
    const allMetrics = await onGetMetrics();
    const metrics = [];
    for (const metric of allMetrics) {
      if (metric.value) {
        metrics.push(metric.value);
      }
    }
    return metrics;
  }, [onGetMetrics]);
  const asyncSelect = () => {
    return /* @__PURE__ */ jsxs(InputGroup, { children: [
      /* @__PURE__ */ jsx(
        Combobox,
        {
          placeholder: "Select metric",
          width: "auto",
          minWidth: 25,
          options: loadOptions,
          value: query.metric,
          onChange: onComboboxChange,
          createCustomValue: true
        }
      ),
      /* @__PURE__ */ jsx(
        Button,
        {
          tooltip: "Open metrics explorer",
          "aria-label": "Open metrics explorer",
          variant: "secondary",
          icon: "book-open",
          onClick: () => {
            tracking("grafana_prometheus_metric_encyclopedia_open", null, "", query);
            setMetricsModalOpen(true);
          }
        }
      )
    ] });
  };
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    !datasource.lookupsDisabled && metricsModalOpen && /* @__PURE__ */ jsx(
      MetricsModal,
      {
        datasource,
        isOpen: metricsModalOpen,
        onClose: () => setMetricsModalOpen(false),
        query,
        onChange,
        initialMetrics: loadMetricsExplorerMetrics
      }
    ),
    variableEditor ? /* @__PURE__ */ jsx(InlineFieldRow, { children: /* @__PURE__ */ jsx(
      InlineField,
      {
        label: "Metric",
        labelWidth: 20,
        tooltip: /* @__PURE__ */ jsx("div", { children: "Optional: returns a list of label values for the label name in the specified metric." }),
        children: asyncSelect()
      }
    ) }) : /* @__PURE__ */ jsx(EditorFieldGroup, { children: /* @__PURE__ */ jsx(EditorField, { label: "Metric", children: asyncSelect() }) })
  ] });
}
const formatPrometheusLabelFiltersToString = (queryString, labelsFilters) => {
  const filterArray = labelsFilters ? formatPrometheusLabelFilters(labelsFilters) : [];
  return `label_values({__name__=~".*${queryString}"${filterArray ? filterArray.join("") : ""}},__name__)`;
};
const formatPrometheusLabelFilters = (labelsFilters) => {
  return labelsFilters.map((label) => {
    return `,${label.label}="${label.value}"`;
  });
};
const formatKeyValueStringsForLabelValuesQuery = (query, labelsFilters) => {
  const queryString = regexifyLabelValuesQueryString(query);
  return formatPrometheusLabelFiltersToString(queryString, labelsFilters);
};

export { MetricCombobox, formatPrometheusLabelFilters, formatPrometheusLabelFiltersToString };
//# sourceMappingURL=MetricCombobox.mjs.map
