import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { useState } from 'react';
import { selectors } from '@grafana/e2e-selectors';
import { DataSourcePicker, config } from '@grafana/runtime';
import { useTheme2, InlineField, Switch, Input, Button } from '@grafana/ui';
import { overhaulStyles, PROM_CONFIG_LABEL_WIDTH, docsTip } from './ConfigEditor.mjs';

function ExemplarSetting({ value, onChange, onDelete, disabled }) {
  const [isInternalLink, setIsInternalLink] = useState(Boolean(value.datasourceUid));
  const theme = useTheme2();
  const styles = overhaulStyles(theme);
  return /* @__PURE__ */ jsxs("div", { className: "gf-form-group", children: [
    /* @__PURE__ */ jsx(
      InlineField,
      {
        label: "Internal link",
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        disabled,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          "Enable this option if you have an internal link. When enabled, this reveals the data source selector. Select the backend tracing data store for your exemplar data. ",
          docsTip()
        ] }),
        interactive: true,
        className: styles.switchField,
        children: /* @__PURE__ */ jsx(Fragment, { children: /* @__PURE__ */ jsx(
          Switch,
          {
            value: isInternalLink,
            "data-testid": selectors.components.DataSource.Prometheus.configPage.internalLinkSwitch,
            onChange: (ev) => setIsInternalLink(ev.currentTarget.checked)
          }
        ) })
      }
    ),
    isInternalLink ? /* @__PURE__ */ jsx(
      InlineField,
      {
        label: "Data source",
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          "The data source the exemplar is going to navigate to. ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          DataSourcePicker,
          {
            filter: config.featureToggles.azureMonitorPrometheusExemplars ? undefined : (ds) => ds.type !== "grafana-azure-monitor-datasource",
            tracing: true,
            current: value.datasourceUid,
            noDefault: true,
            width: 40,
            onChange: (ds) => onChange({
              ...value,
              datasourceUid: ds.uid,
              url: undefined
            })
          }
        )
      }
    ) : /* @__PURE__ */ jsx(
      InlineField,
      {
        label: "URL",
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          "The URL of the trace backend the user would go to see its trace. ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          Input,
          {
            placeholder: "https://example.com/${__value.raw}",
            spellCheck: false,
            width: 40,
            value: value.url,
            onChange: (event) => onChange({
              ...value,
              datasourceUid: undefined,
              url: event.currentTarget.value
            })
          }
        )
      }
    ),
    /* @__PURE__ */ jsx(
      InlineField,
      {
        label: "URL Label",
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          "Use to override the button label on the exemplar traceID field. ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          Input,
          {
            placeholder: "Go to example.com",
            spellCheck: false,
            width: 40,
            value: value.urlDisplayLabel,
            onChange: (event) => onChange({
              ...value,
              urlDisplayLabel: event.currentTarget.value
            })
          }
        )
      }
    ),
    /* @__PURE__ */ jsx(
      InlineField,
      {
        label: "Label name",
        labelWidth: PROM_CONFIG_LABEL_WIDTH,
        tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
          "The name of the field in the labels object that should be used to get the traceID. ",
          docsTip()
        ] }),
        disabled,
        interactive: true,
        children: /* @__PURE__ */ jsx(
          Input,
          {
            placeholder: "traceID",
            spellCheck: false,
            width: 40,
            value: value.name,
            onChange: (event) => onChange({
              ...value,
              name: event.currentTarget.value
            })
          }
        )
      }
    ),
    !disabled && /* @__PURE__ */ jsx(InlineField, { label: "Remove exemplar link", labelWidth: PROM_CONFIG_LABEL_WIDTH, disabled, children: /* @__PURE__ */ jsx(
      Button,
      {
        variant: "destructive",
        title: "Remove exemplar link",
        icon: "times",
        onClick: (event) => {
          event.preventDefault();
          onDelete();
        }
      }
    ) })
  ] });
}

export { ExemplarSetting };
//# sourceMappingURL=ExemplarSetting.mjs.map
