import { cloneDeep } from 'lodash';
import { of } from 'rxjs';
import { mergeMap, map } from 'rxjs/operators';
import { getFrameMatchers } from './matchers.mjs';
import { standardTransformersRegistry } from './standardTransformersRegistry.mjs';

const getOperator = (config, ctx) => (source) => {
  var _a, _b;
  const info = standardTransformersRegistry.get(config.id);
  if (!info) {
    return source;
  }
  const defaultOptions = (_a = info.transformation.defaultOptions) != null ? _a : {};
  const options = { ...defaultOptions, ...config.options };
  const isScenes = window.__grafanaSceneContext != null;
  const interpolated = isScenes ? options : deepIterate(cloneDeep(options), (v) => {
    if (typeof v === "string") {
      return ctx.interpolate(v);
    }
    return v;
  });
  const matcher = ((_b = config.filter) == null ? undefined : _b.options) ? getFrameMatchers(config.filter) : undefined;
  return source.pipe(
    mergeMap(
      (before) => of(filterInput(before, matcher)).pipe(
        info.transformation.operator(interpolated, ctx),
        postProcessTransform(before, info, matcher)
      )
    )
  );
};
function filterInput(data, matcher) {
  if (matcher) {
    return data.filter((v) => matcher(v));
  }
  return data;
}
const postProcessTransform = (before, info, matcher) => (source) => source.pipe(
  map((after) => {
    if (after === before) {
      return after;
    }
    if (matcher) {
      let insert = 0;
      const append = before.filter((v, idx) => {
        const keep = !matcher(v);
        if (keep && !insert) {
          insert = idx;
        }
        return keep;
      });
      if (append.length) {
        after.splice(insert, 0, ...append);
      }
    }
    return after;
  })
);
function transformDataFrame(options, data, ctx) {
  const stream = of(data);
  if (!options.length) {
    return stream;
  }
  const operators = [];
  const context = ctx != null ? ctx : { interpolate: (str) => str };
  for (let index = 0; index < options.length; index++) {
    const config = options[index];
    if (isCustomTransformation(config)) {
      operators.push(config(context));
    } else {
      if (config.disabled) {
        continue;
      }
      operators.push(getOperator(config, context));
    }
  }
  return stream.pipe.apply(stream, operators);
}
function isCustomTransformation(t) {
  return typeof t === "function";
}
function deepIterate(obj, doSomething) {
  var _a;
  if (Array.isArray(obj)) {
    return obj.map((o) => deepIterate(o, doSomething));
  }
  if (typeof obj === "object") {
    for (const key in obj) {
      obj[key] = deepIterate(obj[key], doSomething);
    }
    return obj;
  } else {
    return (_a = doSomething(obj)) != null ? _a : obj;
  }
}

export { transformDataFrame };
//# sourceMappingURL=transformDataFrame.mjs.map
