import { map } from 'rxjs/operators';
import { getFieldDisplayName } from '../../field/fieldState.mjs';
import { SpecialValue, TransformationApplicabilityLevels } from '../../types/transformations.mjs';
import { fieldMatchers } from '../matchers.mjs';
import { FieldMatcherID } from '../matchers/ids.mjs';
import { DataTransformerID } from './ids.mjs';

var _a, _b, _c;
const DEFAULT_COLUMN_FIELD = "Time";
const DEFAULT_ROW_FIELD = "Time";
const DEFAULT_VALUE_FIELD = "Value";
const DEFAULT_EMPTY_VALUE = SpecialValue.Empty;
const supportDataplaneFallback = (_c = (_b = (_a = window == null ? undefined : window.grafanaBootData) == null ? undefined : _a.settings) == null ? undefined : _b.featureToggles) == null ? undefined : _c.dataplaneFrontendFallback;
const groupingToMatrixTransformer = {
  id: DataTransformerID.groupingToMatrix,
  name: "Grouping to Matrix",
  description: "Groups series by field and return a matrix visualisation",
  defaultOptions: {
    columnField: DEFAULT_COLUMN_FIELD,
    rowField: DEFAULT_ROW_FIELD,
    valueField: DEFAULT_VALUE_FIELD
  },
  /**
   * Grouping to matrix requires at least 3 fields to work.
   */
  isApplicable: (data) => {
    let numFields = 0;
    for (const frame of data) {
      numFields += frame.fields.length;
    }
    return numFields >= 3 ? TransformationApplicabilityLevels.Applicable : TransformationApplicabilityLevels.NotApplicable;
  },
  isApplicableDescription: (data) => {
    let numFields = 0;
    for (const frame of data) {
      numFields += frame.fields.length;
    }
    return `Grouping to matrix requiers at least 3 fields to work. Currently there are ${numFields} fields.`;
  },
  operator: (options, ctx) => (source) => source.pipe(
    map((data) => {
      var _a2, _b2;
      const columnFieldMatch = options.columnField || DEFAULT_COLUMN_FIELD;
      const rowFieldMatch = options.rowField || DEFAULT_ROW_FIELD;
      const valueFieldMatch = options.valueField || DEFAULT_VALUE_FIELD;
      const emptyValue = options.emptyValue || DEFAULT_EMPTY_VALUE;
      if (data.length !== 1) {
        return data;
      }
      const frame = data[0];
      const keyColumnField = findKeyField(frame, columnFieldMatch);
      const keyRowField = findKeyField(frame, rowFieldMatch);
      const valueField = findKeyField(frame, valueFieldMatch);
      const rowColumnField = `${rowFieldMatch}\\${columnFieldMatch}`;
      if (!keyColumnField || !keyRowField || !valueField) {
        return data;
      }
      const columnValues = uniqueValues(keyColumnField.values);
      const rowValues = uniqueValues(keyRowField.values);
      const matrixValues = {};
      for (let index = 0; index < valueField.values.length; index++) {
        const columnName = keyColumnField.values[index];
        const rowName = keyRowField.values[index];
        const value = valueField.values[index];
        if (!matrixValues[columnName]) {
          matrixValues[columnName] = {};
        }
        matrixValues[columnName][rowName] = value;
      }
      const fields = [
        {
          name: rowColumnField,
          values: rowValues,
          type: keyRowField.type,
          config: {}
        }
      ];
      for (const columnName of columnValues) {
        let values = [];
        for (const rowName of rowValues) {
          const value = (_a2 = matrixValues[columnName][rowName]) != null ? _a2 : getSpecialValue(emptyValue);
          values.push(value);
        }
        if (supportDataplaneFallback && typeof columnName === "number") {
          valueField.config = { ...valueField.config, displayNameFromDS: undefined };
        }
        fields.push({
          name: (_b2 = columnName == null ? undefined : columnName.toString()) != null ? _b2 : null,
          values,
          config: valueField.config,
          type: valueField.type
        });
      }
      return [
        {
          fields,
          length: rowValues.length
        }
      ];
    })
  )
};
function uniqueValues(values) {
  const unique = new Set(values);
  return Array.from(unique);
}
function findKeyField(frame, matchTitle) {
  for (let fieldIndex = 0; fieldIndex < frame.fields.length; fieldIndex++) {
    const field = frame.fields[fieldIndex];
    let matches;
    if (supportDataplaneFallback) {
      const matcher = fieldMatchers.get(FieldMatcherID.byName).get(matchTitle);
      matches = matcher(field, frame, [frame]);
    } else {
      matches = matchTitle === getFieldDisplayName(field);
    }
    if (matches) {
      return field;
    }
  }
  return null;
}
function getSpecialValue(specialValue) {
  switch (specialValue) {
    case SpecialValue.False:
      return false;
    case SpecialValue.True:
      return true;
    case SpecialValue.Null:
      return null;
    case SpecialValue.Zero:
      return 0;
    case SpecialValue.Empty:
    default:
      return "";
  }
}

export { groupingToMatrixTransformer };
//# sourceMappingURL=groupingToMatrix.mjs.map
