import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import * as React from 'react';
import { CoreApp } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { EditorRow, EditorField, EditorSwitch } from '@grafana/plugin-ui';
import { AutoSizeInput, Select, RadioButtonGroup } from '@grafana/ui';
import { getQueryTypeOptions, getQueryTypeChangeHandler } from '../../components/PromExploreExtraField.mjs';
import { QueryOptionGroup } from '../shared/QueryOptionGroup.mjs';
import { FORMAT_OPTIONS, INTERVAL_FACTOR_OPTIONS } from './PromQueryEditorSelector.mjs';
import { PromQueryLegendEditor, getLegendModeLabel } from './PromQueryLegendEditor.mjs';

const PromQueryBuilderOptions = React.memo(
  ({ query, app, onChange, onRunQuery }) => {
    const onChangeFormat = (value) => {
      onChange({ ...query, format: value.value });
      onRunQuery();
    };
    const onChangeStep = (evt) => {
      onChange({ ...query, interval: evt.currentTarget.value.trim() });
      onRunQuery();
    };
    const queryTypeOptions = getQueryTypeOptions(
      app === CoreApp.Explore || app === CoreApp.Correlations || app === CoreApp.PanelEditor
    );
    const onQueryTypeChange = getQueryTypeChangeHandler(query, onChange);
    const onExemplarChange = (event) => {
      const isEnabled = event.currentTarget.checked;
      onChange({ ...query, exemplar: isEnabled });
      onRunQuery();
    };
    const onIntervalFactorChange = (value) => {
      onChange({ ...query, intervalFactor: value.value });
      onRunQuery();
    };
    const formatOption = FORMAT_OPTIONS.find((option) => option.value === query.format) || FORMAT_OPTIONS[0];
    const queryTypeValue = getQueryTypeValue(query);
    const queryTypeLabel = queryTypeOptions.find((x) => x.value === queryTypeValue).label;
    return /* @__PURE__ */ jsx(EditorRow, { children: /* @__PURE__ */ jsx("div", { "data-testid": selectors.components.DataSource.Prometheus.queryEditor.options, children: /* @__PURE__ */ jsxs(
      QueryOptionGroup,
      {
        title: "Options",
        collapsedInfo: getCollapsedInfo(query, formatOption.label, queryTypeLabel, app),
        children: [
          /* @__PURE__ */ jsx(
            PromQueryLegendEditor,
            {
              legendFormat: query.legendFormat,
              onChange: (legendFormat) => onChange({ ...query, legendFormat }),
              onRunQuery
            }
          ),
          /* @__PURE__ */ jsx(
            EditorField,
            {
              label: "Min step",
              tooltip: /* @__PURE__ */ jsxs(Fragment, { children: [
                "An additional lower limit for the step parameter of the Prometheus query and for the",
                " ",
                /* @__PURE__ */ jsx("code", { children: "$__interval" }),
                " and ",
                /* @__PURE__ */ jsx("code", { children: "$__rate_interval" }),
                " variables."
              ] }),
              children: /* @__PURE__ */ jsx(
                AutoSizeInput,
                {
                  type: "text",
                  "aria-label": "Set lower limit for the step parameter",
                  placeholder: "auto",
                  minWidth: 10,
                  onCommitChange: onChangeStep,
                  defaultValue: query.interval,
                  "data-test-id": "prometheus-step"
                }
              )
            }
          ),
          /* @__PURE__ */ jsx(EditorField, { label: "Format", children: /* @__PURE__ */ jsx(
            Select,
            {
              "data-testid": selectors.components.DataSource.Prometheus.queryEditor.format,
              value: formatOption,
              allowCustomValue: true,
              onChange: onChangeFormat,
              options: FORMAT_OPTIONS
            }
          ) }),
          /* @__PURE__ */ jsx(EditorField, { label: "Type", "data-testid": selectors.components.DataSource.Prometheus.queryEditor.type, children: /* @__PURE__ */ jsx(RadioButtonGroup, { options: queryTypeOptions, value: queryTypeValue, onChange: onQueryTypeChange }) }),
          shouldShowExemplarSwitch(query, app) && /* @__PURE__ */ jsx(EditorField, { label: "Exemplars", children: /* @__PURE__ */ jsx(
            EditorSwitch,
            {
              value: query.exemplar || false,
              onChange: onExemplarChange,
              "data-test-id": "prometheus-exemplars"
            }
          ) }),
          query.intervalFactor && query.intervalFactor > 1 && /* @__PURE__ */ jsx(EditorField, { label: "Resolution", children: /* @__PURE__ */ jsx(
            Select,
            {
              "aria-label": "Select resolution",
              isSearchable: false,
              options: INTERVAL_FACTOR_OPTIONS,
              onChange: onIntervalFactorChange,
              value: INTERVAL_FACTOR_OPTIONS.find((option) => option.value === query.intervalFactor)
            }
          ) })
        ]
      }
    ) }) });
  }
);
function shouldShowExemplarSwitch(query, app) {
  if (app === CoreApp.UnifiedAlerting || !query.range) {
    return false;
  }
  return true;
}
function getQueryTypeValue(query) {
  return query.range && query.instant ? "both" : query.instant ? "instant" : "range";
}
function getCollapsedInfo(query, formatOption, queryType, app) {
  var _a;
  const items = [];
  items.push(`Legend: ${getLegendModeLabel(query.legendFormat)}`);
  items.push(`Format: ${formatOption}`);
  items.push(`Step: ${(_a = query.interval) != null ? _a : "auto"}`);
  items.push(`Type: ${queryType}`);
  if (shouldShowExemplarSwitch(query, app)) {
    if (query.exemplar) {
      items.push(`Exemplars: true`);
    } else {
      items.push(`Exemplars: false`);
    }
  }
  return items;
}
PromQueryBuilderOptions.displayName = "PromQueryBuilderOptions";

export { PromQueryBuilderOptions };
//# sourceMappingURL=PromQueryBuilderOptions.mjs.map
