import { getCompletions } from './completions.mjs';
import { getSituation } from './situation.mjs';
import { NeverCaseError } from './util.mjs';

function getSuggestOptions() {
  return {
    // monaco-editor sometimes provides suggestions automatically, i am not
    // sure based on what, seems to be by analyzing the words already
    // written.
    // to try it out:
    // - enter `go_goroutines{job~`
    // - have the cursor at the end of the string
    // - press ctrl-enter
    // - you will get two suggestions
    // those were not provided by grafana, they are offered automatically.
    // i want to remove those. the only way i found is:
    // - every suggestion-item has a `kind` attribute,
    //   that controls the icon to the left of the suggestion.
    // - items auto-generated by monaco have `kind` set to `text`.
    // - we make sure grafana-provided suggestions do not have `kind` set to `text`.
    // - and then we tell monaco not to show suggestions of kind `text`
    showWords: false
  };
}
function getMonacoCompletionItemKind(type, monaco) {
  switch (type) {
    case "DURATION":
      return monaco.languages.CompletionItemKind.Unit;
    case "FUNCTION":
      return monaco.languages.CompletionItemKind.Variable;
    case "HISTORY":
      return monaco.languages.CompletionItemKind.Snippet;
    case "LABEL_NAME":
      return monaco.languages.CompletionItemKind.Enum;
    case "LABEL_VALUE":
      return monaco.languages.CompletionItemKind.EnumMember;
    case "METRIC_NAME":
      return monaco.languages.CompletionItemKind.Constructor;
    default:
      throw new NeverCaseError(type);
  }
}
function getCompletionProvider(monaco, dataProvider) {
  const provideCompletionItems = (model, position) => {
    var _a;
    const word = model.getWordAtPosition(position);
    const range = word != null ? monaco.Range.lift({
      startLineNumber: position.lineNumber,
      endLineNumber: position.lineNumber,
      startColumn: word.startColumn,
      endColumn: word.endColumn
    }) : monaco.Range.fromPositions(position);
    const positionClone = {
      column: position.column,
      lineNumber: position.lineNumber
    };
    dataProvider.monacoSettings.setInputInRange(model.getValueInRange(range));
    if (window.getSelection) {
      const selectedText = (_a = window.getSelection()) == null ? undefined : _a.toString();
      if (selectedText && selectedText.length > 0) {
        positionClone.column = positionClone.column - selectedText.length;
      }
    }
    const offset = model.getOffsetAt(positionClone);
    const situation = getSituation(model.getValue(), offset);
    const completionsPromise = situation != null ? getCompletions(situation, dataProvider) : Promise.resolve([]);
    return completionsPromise.then((items) => {
      const maxIndexDigits = items.length.toString().length;
      const suggestions = items.map((item, index) => ({
        kind: getMonacoCompletionItemKind(item.type, monaco),
        label: item.label,
        insertText: item.insertText,
        insertTextRules: item.insertTextRules,
        detail: item.detail,
        documentation: item.documentation,
        sortText: index.toString().padStart(maxIndexDigits, "0"),
        // to force the order we have
        range,
        command: item.triggerOnInsert ? {
          id: "editor.action.triggerSuggest",
          title: ""
        } : undefined
      }));
      return { suggestions, incomplete: dataProvider.monacoSettings.suggestionsIncomplete };
    });
  };
  return {
    triggerCharacters: ["{", ",", "[", "(", "=", "~", " ", '"'],
    provideCompletionItems
  };
}

export { getCompletionProvider, getSuggestOptions };
//# sourceMappingURL=index.mjs.map
