import { isNumber } from 'lodash';
import { reduceField, ReducerID } from '../transformations/fieldReducer.mjs';
import { FieldType } from '../types/dataFrame.mjs';
import { getFieldColorModeForField } from './fieldColor.mjs';
import { getActiveThresholdForValue } from './thresholds.mjs';

function getScaleCalculator(field, theme) {
  var _a, _b;
  if (field.type === FieldType.boolean) {
    return getBooleanScaleCalculator(field, theme);
  }
  const mode = getFieldColorModeForField(field);
  const getColor = mode.getCalculator(field, theme);
  const info = (_b = (_a = field.state) == null ? undefined : _a.range) != null ? _b : getMinMaxAndDelta(field);
  return (value) => {
    let percent = 0;
    if (value !== -Infinity) {
      percent = (value - info.min) / info.delta;
      if (Number.isNaN(percent)) {
        percent = 0;
      }
    }
    const threshold = getActiveThresholdForValue(field, value, percent);
    return {
      percent,
      threshold,
      color: getColor(value, percent, threshold)
    };
  };
}
function getBooleanScaleCalculator(field, theme) {
  const trueValue = {
    color: theme.visualization.getColorByName("green"),
    percent: 1,
    threshold: undefined
  };
  const falseValue = {
    color: theme.visualization.getColorByName("red"),
    percent: 0,
    threshold: undefined
  };
  const mode = getFieldColorModeForField(field);
  if (mode.isContinuous && mode.getColors) {
    const colors = mode.getColors(theme);
    trueValue.color = colors[colors.length - 1];
    falseValue.color = colors[0];
  }
  return (value) => {
    return Boolean(value) ? trueValue : falseValue;
  };
}
function getMinMaxAndDelta(field) {
  if (field.type !== FieldType.number) {
    return { min: 0, max: 100, delta: 100 };
  }
  let min = field.config.min;
  let max = field.config.max;
  if (!isNumber(min) || !isNumber(max)) {
    if (field.values && field.values.length) {
      const stats = reduceField({ field, reducers: [ReducerID.min, ReducerID.max] });
      if (!isNumber(min)) {
        min = stats[ReducerID.min];
      }
      if (!isNumber(max)) {
        max = stats[ReducerID.max];
      }
    } else {
      min = 0;
      max = 100;
    }
  }
  return {
    min,
    max,
    delta: max - min
  };
}
function getFieldConfigWithMinMax(field, local) {
  var _a;
  const { config } = field;
  let { min, max } = config;
  if (isNumber(min) && isNumber(max)) {
    return config;
  }
  if (local || !((_a = field.state) == null ? undefined : _a.range)) {
    return { ...config, ...getMinMaxAndDelta(field) };
  }
  return { ...config, ...field.state.range };
}

export { getFieldConfigWithMinMax, getMinMaxAndDelta, getScaleCalculator };
//# sourceMappingURL=scale.mjs.map
