import { jsxs, jsx } from 'react/jsx-runtime';
import { cx, css } from '@emotion/css';
import * as React from 'react';
import { useState, useCallback, useRef, useImperativeHandle } from 'react';
import '@grafana/data';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import 'micro-memoize';
import '@emotion/react';
import { getFocusStyles } from '../../themes/mixins.mjs';
import '../../utils/skeleton.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { Stack } from '../Layout/Stack/Stack.mjs';
import { SubMenu } from './SubMenu.mjs';

const MenuItem = React.memo(
  React.forwardRef((props, ref) => {
    const {
      url,
      icon,
      label,
      description,
      ariaLabel,
      ariaChecked,
      target,
      onClick,
      className,
      active,
      disabled,
      destructive,
      childItems,
      role,
      tabIndex = -1,
      customSubMenuContainerStyles,
      shortcut,
      testId
    } = props;
    const styles = useStyles2(getStyles);
    const [isActive, setIsActive] = useState(active);
    const [isSubMenuOpen, setIsSubMenuOpen] = useState(false);
    const onMouseEnter = useCallback(() => {
      if (disabled) {
        return;
      }
      setIsSubMenuOpen(true);
      setIsActive(true);
    }, [disabled]);
    const onMouseLeave = useCallback(() => {
      if (disabled) {
        return;
      }
      setIsSubMenuOpen(false);
      setIsActive(false);
    }, [disabled]);
    const hasSubMenu = childItems && childItems.length > 0;
    const ItemElement = hasSubMenu ? "div" : url === undefined ? "button" : "a";
    const itemStyle = cx(
      {
        [styles.item]: true,
        [styles.active]: isActive,
        [styles.disabled]: disabled,
        [styles.destructive]: destructive && !disabled
      },
      className
    );
    const disabledProps = {
      [ItemElement === "button" ? "disabled" : "aria-disabled"]: disabled,
      ...ItemElement === "a" && disabled && { href: undefined, onClick: undefined },
      ...disabled && {
        tabIndex: -1,
        ["data-disabled"]: disabled
        // used to identify disabled items in Menu.tsx
      }
    };
    const localRef = useRef(null);
    useImperativeHandle(ref, () => localRef.current);
    const handleKeys = (event) => {
      switch (event.key) {
        case "ArrowRight":
          event.preventDefault();
          event.stopPropagation();
          if (hasSubMenu) {
            setIsSubMenuOpen(true);
            setIsActive(true);
          }
          break;
      }
    };
    const closeSubMenu = () => {
      var _a;
      setIsSubMenuOpen(false);
      setIsActive(false);
      (_a = localRef == null ? undefined : localRef.current) == null ? undefined : _a.focus();
    };
    const hasShortcut = Boolean(shortcut && shortcut.length > 0);
    return /* @__PURE__ */ jsxs(
      ItemElement,
      {
        target,
        className: itemStyle,
        rel: target === "_blank" ? "noopener noreferrer" : undefined,
        href: url,
        onClick: (event) => {
          if (hasSubMenu && !isSubMenuOpen) {
            event.preventDefault();
            event.stopPropagation();
          }
          onClick == null ? undefined : onClick(event);
        },
        onMouseEnter,
        onMouseLeave,
        onKeyDown: handleKeys,
        role: !url ? role || "menuitem" : role,
        "data-role": "menuitem",
        ref: localRef,
        "data-testid": testId,
        "aria-label": ariaLabel,
        "aria-checked": ariaChecked,
        tabIndex,
        ...disabledProps,
        children: [
          /* @__PURE__ */ jsxs(Stack, { direction: "row", justifyContent: "flex-start", alignItems: "center", children: [
            icon && /* @__PURE__ */ jsx(Icon, { name: icon, className: styles.icon, "aria-hidden": true }),
            /* @__PURE__ */ jsx("span", { className: styles.ellipsis, children: label }),
            /* @__PURE__ */ jsxs("div", { className: cx(styles.rightWrapper, { [styles.withShortcut]: hasShortcut }), children: [
              hasShortcut && /* @__PURE__ */ jsxs("div", { className: styles.shortcut, children: [
                /* @__PURE__ */ jsx(Icon, { name: "keyboard", title: "keyboard shortcut" }),
                shortcut
              ] }),
              hasSubMenu && /* @__PURE__ */ jsx(
                SubMenu,
                {
                  items: childItems,
                  isOpen: isSubMenuOpen,
                  close: closeSubMenu,
                  customStyle: customSubMenuContainerStyles
                }
              )
            ] })
          ] }),
          description && /* @__PURE__ */ jsx(
            "div",
            {
              className: cx(styles.description, styles.ellipsis, {
                [styles.descriptionWithIcon]: icon !== undefined
              }),
              children: description
            }
          ),
          props.component ? /* @__PURE__ */ jsx(props.component, {}) : null
        ]
      }
    );
  })
);
MenuItem.displayName = "MenuItem";
const getStyles = (theme) => {
  return {
    item: css({
      background: "none",
      cursor: "pointer",
      whiteSpace: "nowrap",
      color: theme.colors.text.primary,
      display: "flex",
      flexDirection: "column",
      alignItems: "stretch",
      justifyContent: "center",
      padding: theme.spacing(0.5, 1.5),
      minHeight: theme.spacing(4),
      borderRadius: theme.shape.radius.default,
      margin: 0,
      border: "none",
      width: "100%",
      position: "relative",
      "&:hover, &:focus-visible": {
        background: theme.colors.action.hover,
        color: theme.colors.text.primary,
        textDecoration: "none"
      },
      "&:focus-visible": getFocusStyles(theme)
    }),
    active: css({
      background: theme.colors.action.hover
    }),
    destructive: css({
      color: theme.colors.error.text,
      svg: {
        color: theme.colors.error.text
      },
      "&:hover, &:focus, &:focus-visible": {
        background: theme.colors.error.main,
        color: theme.colors.error.contrastText,
        svg: {
          color: theme.colors.error.contrastText
        }
      }
    }),
    disabled: css({
      color: theme.colors.action.disabledText,
      label: "menu-item-disabled",
      "&:hover, &:focus, &:focus-visible": {
        cursor: "not-allowed",
        background: "none",
        color: theme.colors.action.disabledText
      }
    }),
    icon: css({
      opacity: 0.7,
      color: theme.colors.text.secondary
    }),
    rightWrapper: css({
      display: "flex",
      alignItems: "center",
      marginLeft: "auto"
    }),
    withShortcut: css({
      minWidth: theme.spacing(10.5)
    }),
    shortcut: css({
      display: "flex",
      alignItems: "center",
      gap: theme.spacing(1),
      marginLeft: theme.spacing(2),
      color: theme.colors.text.secondary,
      opacity: 0.7
    }),
    description: css({
      ...theme.typography.bodySmall,
      color: theme.colors.text.secondary,
      textAlign: "start"
    }),
    descriptionWithIcon: css({
      marginLeft: theme.spacing(3)
    }),
    ellipsis: css({
      overflow: "hidden",
      textOverflow: "ellipsis",
      whiteSpace: "nowrap"
    })
  };
};

export { MenuItem };
//# sourceMappingURL=MenuItem.mjs.map
