"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadJestPluginConfig = exports.jestConfig = exports.allowedJestConfigOverrides = void 0;
const tslib_1 = require("tslib");
const fs_1 = tslib_1.__importDefault(require("fs"));
const path = require("path");
exports.allowedJestConfigOverrides = [
    'snapshotSerializers',
    'moduleNameMapper',
    'globalSetup',
    'globalTeardown',
    'testEnvironment',
];
const getSetupFile = (filePath) => {
    if (fs_1.default.existsSync(`${filePath}.js`)) {
        return `${filePath}.js`;
    }
    if (fs_1.default.existsSync(`${filePath}.ts`)) {
        return `${filePath}.ts`;
    }
    return undefined;
};
const jestConfig = (baseDir = process.cwd()) => {
    const jestConfigOverrides = (require(path.resolve(baseDir, 'package.json')).jest || {});
    const deniedOverrides = jestConfigOverrides
        ? Object.keys(jestConfigOverrides).filter((override) => exports.allowedJestConfigOverrides.indexOf(override) === -1)
        : [];
    if (deniedOverrides.length > 0) {
        console.error("\ngrafana-toolkit doesn't support following Jest options: ", deniedOverrides);
        console.log('Supported Jest options are: ', JSON.stringify(exports.allowedJestConfigOverrides));
        throw new Error('Provided Jest config is not supported');
    }
    const shimsFilePath = path.resolve(baseDir, 'config/jest-shim');
    const setupFilePath = path.resolve(baseDir, 'config/jest-setup');
    // Mock css imports for tests. Otherwise Jest will have troubles understanding SASS/CSS imports
    const { moduleNameMapper } = jestConfigOverrides, otherOverrides = tslib_1.__rest(jestConfigOverrides, ["moduleNameMapper"]);
    const moduleNameMapperConfig = Object.assign({ '\\.(css|sass|scss)$': `${__dirname}/styles.mock.js`, 'react-inlinesvg': `${__dirname}/react-inlinesvg.tsx` }, moduleNameMapper);
    const setupFile = getSetupFile(setupFilePath);
    const shimsFile = getSetupFile(shimsFilePath);
    const setupFiles = [setupFile, shimsFile, `${__dirname}/matchMedia.js`, require.resolve('jest-canvas-mock')].filter((f) => f);
    const defaultJestConfig = {
        verbose: false,
        moduleDirectories: ['node_modules', 'src'],
        moduleFileExtensions: ['ts', 'tsx', 'js', 'jsx', 'json'],
        setupFiles,
        globals: {
            'ts-jest': {
                isolatedModules: true,
                tsconfig: path.resolve(baseDir, 'tsconfig.json'),
            },
        },
        coverageReporters: ['json-summary', 'text', 'lcov'],
        collectCoverageFrom: ['src/**/*.{ts,tsx}', '!**/node_modules/**', '!**/vendor/**'],
        reporters: [
            'default',
            [
                require.resolve('jest-junit'),
                {
                    outputDirectory: 'coverage',
                },
            ],
        ],
        testEnvironment: 'jsdom',
        testMatch: [
            '<rootDir>/src/**/__tests__/**/*.{js,jsx,ts,tsx}',
            '<rootDir>/src/**/*.{spec,test,jest}.{js,jsx,ts,tsx}',
            '<rootDir>/spec/**/*.{spec,test,jest}.{js,jsx,ts,tsx}',
        ],
        transform: {
            '^.+\\.(js|jsx|mjs)$': [
                require.resolve('babel-jest'),
                { configFile: path.resolve(__dirname, './jest.babel.config.js') },
            ],
            '^.+\\.tsx?$': require.resolve('ts-jest'),
        },
        transformIgnorePatterns: [
            '[/\\\\\\\\]node_modules[/\\\\\\\\].+\\\\.(js|jsx|ts|tsx)$',
            '^.+\\\\.module\\\\.(css|sass|scss)$',
        ],
        moduleNameMapper: moduleNameMapperConfig,
    };
    return Object.assign(Object.assign({}, defaultJestConfig), otherOverrides);
};
exports.jestConfig = jestConfig;
/**
 * This will load the existing just setup, or use the default if it exists
 */
const loadJestPluginConfig = (baseDir = process.cwd()) => {
    const cfgpath = path.resolve(baseDir, 'jest.config.js');
    if (!fs_1.default.existsSync(cfgpath)) {
        const toolkitDir = path.dirname(require.resolve(`@grafana/toolkit/package.json`));
        const src = path.join(toolkitDir, 'src/config/jest.plugin.config.local.js');
        fs_1.default.copyFileSync(src, cfgpath);
        console.log('Using standard jest plugin config', src);
    }
    return require(cfgpath);
};
exports.loadJestPluginConfig = loadJestPluginConfig;
//# sourceMappingURL=jest.plugin.config.js.map