import { isNumber } from 'lodash';
import { NullValueMode } from '../types/data.js';
import { FieldType } from '../types/dataFrame.js';
import '@grafana/schema';
import '../datetime/moment_wrapper.js';
import '../types/datasource.js';
import '../types/legacyEvents.js';
import { Registry } from '../utils/Registry.js';

var __defProp = Object.defineProperty;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
var ReducerID = /* @__PURE__ */ ((ReducerID2) => {
  ReducerID2["sum"] = "sum";
  ReducerID2["max"] = "max";
  ReducerID2["min"] = "min";
  ReducerID2["logmin"] = "logmin";
  ReducerID2["mean"] = "mean";
  ReducerID2["variance"] = "variance";
  ReducerID2["stdDev"] = "stdDev";
  ReducerID2["last"] = "last";
  ReducerID2["first"] = "first";
  ReducerID2["count"] = "count";
  ReducerID2["range"] = "range";
  ReducerID2["diff"] = "diff";
  ReducerID2["diffperc"] = "diffperc";
  ReducerID2["delta"] = "delta";
  ReducerID2["step"] = "step";
  ReducerID2["firstNotNull"] = "firstNotNull";
  ReducerID2["lastNotNull"] = "lastNotNull";
  ReducerID2["changeCount"] = "changeCount";
  ReducerID2["distinctCount"] = "distinctCount";
  ReducerID2["allIsZero"] = "allIsZero";
  ReducerID2["allIsNull"] = "allIsNull";
  ReducerID2["allValues"] = "allValues";
  ReducerID2["uniqueValues"] = "uniqueValues";
  return ReducerID2;
})(ReducerID || {});
function reduceField(options) {
  var _a;
  const { field, reducers } = options;
  if (!field || !reducers || reducers.length < 1) {
    return {};
  }
  if ((_a = field.state) == null ? void 0 : _a.calcs) {
    const missing = [];
    for (const s of reducers) {
      if (!field.state.calcs.hasOwnProperty(s)) {
        missing.push(s);
      }
    }
    if (missing.length < 1) {
      return __spreadValues({}, field.state.calcs);
    }
  }
  if (!field.state) {
    field.state = {};
  }
  const queue = fieldReducers.list(reducers);
  const data = field.values;
  if (data.length < 1) {
    const calcs = __spreadValues({}, field.state.calcs);
    for (const reducer of queue) {
      calcs[reducer.id] = reducer.emptyInputResult !== null ? reducer.emptyInputResult : null;
    }
    return field.state.calcs = calcs;
  }
  const { nullValueMode } = field.config;
  const ignoreNulls = nullValueMode === NullValueMode.Ignore;
  const nullAsZero = nullValueMode === NullValueMode.AsZero;
  if (queue.length === 1 && queue[0].reduce) {
    const values2 = queue[0].reduce(field, ignoreNulls, nullAsZero);
    field.state.calcs = __spreadValues(__spreadValues({}, field.state.calcs), values2);
    return values2;
  }
  let values = doStandardCalcs(field, ignoreNulls, nullAsZero);
  for (const reducer of queue) {
    if (!values.hasOwnProperty(reducer.id) && reducer.reduce) {
      values = __spreadValues(__spreadValues({}, values), reducer.reduce(field, ignoreNulls, nullAsZero));
    }
  }
  field.state.calcs = __spreadValues(__spreadValues({}, field.state.calcs), values);
  return values;
}
const fieldReducers = new Registry(() => [
  {
    id: "lastNotNull" /* lastNotNull */,
    name: "Last *",
    description: "Last non-null value",
    standard: true,
    aliasIds: ["current"],
    reduce: calculateLastNotNull
  },
  {
    id: "last" /* last */,
    name: "Last",
    description: "Last value",
    standard: true,
    reduce: calculateLast
  },
  {
    id: "firstNotNull" /* firstNotNull */,
    name: "First *",
    description: "First non-null value",
    standard: true,
    reduce: calculateFirstNotNull
  },
  { id: "first" /* first */, name: "First", description: "First Value", standard: true, reduce: calculateFirst },
  { id: "min" /* min */, name: "Min", description: "Minimum Value", standard: true },
  { id: "max" /* max */, name: "Max", description: "Maximum Value", standard: true },
  { id: "mean" /* mean */, name: "Mean", description: "Average Value", standard: true, aliasIds: ["avg"] },
  {
    id: "variance" /* variance */,
    name: "Variance",
    description: "Variance of all values in a field",
    standard: false,
    reduce: calculateStdDev
  },
  {
    id: "stdDev" /* stdDev */,
    name: "StdDev",
    description: "Standard deviation of all values in a field",
    standard: false,
    reduce: calculateStdDev
  },
  {
    id: "sum" /* sum */,
    name: "Total",
    description: "The sum of all values",
    emptyInputResult: 0,
    standard: true,
    aliasIds: ["total"]
  },
  {
    id: "count" /* count */,
    name: "Count",
    description: "Number of values in response",
    emptyInputResult: 0,
    standard: true
  },
  {
    id: "range" /* range */,
    name: "Range",
    description: "Difference between minimum and maximum values",
    standard: true
  },
  {
    id: "delta" /* delta */,
    name: "Delta",
    description: "Cumulative change in value",
    standard: true
  },
  {
    id: "step" /* step */,
    name: "Step",
    description: "Minimum interval between values",
    standard: true
  },
  {
    id: "diff" /* diff */,
    name: "Difference",
    description: "Difference between first and last values",
    standard: true
  },
  {
    id: "logmin" /* logmin */,
    name: "Min (above zero)",
    description: "Used for log min scale",
    standard: true
  },
  {
    id: "allIsZero" /* allIsZero */,
    name: "All Zeros",
    description: "All values are zero",
    emptyInputResult: false,
    standard: true
  },
  {
    id: "allIsNull" /* allIsNull */,
    name: "All Nulls",
    description: "All values are null",
    emptyInputResult: true,
    standard: true
  },
  {
    id: "changeCount" /* changeCount */,
    name: "Change Count",
    description: "Number of times the value changes",
    standard: false,
    reduce: calculateChangeCount
  },
  {
    id: "distinctCount" /* distinctCount */,
    name: "Distinct Count",
    description: "Number of distinct values",
    standard: false,
    reduce: calculateDistinctCount
  },
  {
    id: "diffperc" /* diffperc */,
    name: "Difference percent",
    description: "Percentage difference between first and last values",
    standard: true
  },
  {
    id: "allValues" /* allValues */,
    name: "All values",
    description: "Returns an array with all values",
    standard: false,
    reduce: (field) => ({ allValues: field.values.toArray() })
  },
  {
    id: "uniqueValues" /* uniqueValues */,
    name: "All unique values",
    description: "Returns an array with all unique values",
    standard: false,
    reduce: (field) => ({
      uniqueValues: [...new Set(field.values.toArray())]
    })
  }
]);
function doStandardCalcs(field, ignoreNulls, nullAsZero) {
  const calcs = {
    sum: 0,
    max: -Number.MAX_VALUE,
    min: Number.MAX_VALUE,
    logmin: Number.MAX_VALUE,
    mean: null,
    last: null,
    first: null,
    lastNotNull: null,
    firstNotNull: null,
    count: 0,
    nonNullCount: 0,
    allIsNull: true,
    allIsZero: true,
    range: null,
    diff: null,
    delta: 0,
    step: Number.MAX_VALUE,
    diffperc: 0,
    // Just used for calculations -- not exposed as a stat
    previousDeltaUp: true
  };
  const data = field.values;
  calcs.count = ignoreNulls ? data.length : data.toArray().filter((val) => val != null).length;
  const isNumberField = field.type === FieldType.number || FieldType.time;
  for (let i = 0; i < data.length; i++) {
    let currentValue = data.get(i);
    if (i === 0) {
      calcs.first = currentValue;
    }
    calcs.last = currentValue;
    if (currentValue === null) {
      if (ignoreNulls) {
        continue;
      }
      if (nullAsZero) {
        currentValue = 0;
      }
    }
    if (currentValue != null) {
      const isFirst = calcs.firstNotNull === null;
      if (isFirst) {
        calcs.firstNotNull = currentValue;
      }
      if (isNumberField) {
        calcs.sum += currentValue;
        calcs.allIsNull = false;
        calcs.nonNullCount++;
        if (!isFirst) {
          const step = currentValue - calcs.lastNotNull;
          if (calcs.step > step) {
            calcs.step = step;
          }
          if (calcs.lastNotNull > currentValue) {
            calcs.previousDeltaUp = false;
            if (i === data.length - 1) {
              calcs.delta += currentValue;
            }
          } else {
            if (calcs.previousDeltaUp) {
              calcs.delta += step;
            } else {
              calcs.delta += currentValue;
            }
            calcs.previousDeltaUp = true;
          }
        }
        if (currentValue > calcs.max) {
          calcs.max = currentValue;
        }
        if (currentValue < calcs.min) {
          calcs.min = currentValue;
        }
        if (currentValue < calcs.logmin && currentValue > 0) {
          calcs.logmin = currentValue;
        }
      }
      if (currentValue !== 0) {
        calcs.allIsZero = false;
      }
      calcs.lastNotNull = currentValue;
    }
  }
  if (calcs.max === -Number.MAX_VALUE) {
    calcs.max = null;
  }
  if (calcs.min === Number.MAX_VALUE) {
    calcs.min = null;
  }
  if (calcs.step === Number.MAX_VALUE) {
    calcs.step = null;
  }
  if (calcs.nonNullCount > 0) {
    calcs.mean = calcs.sum / calcs.nonNullCount;
  }
  if (calcs.allIsNull) {
    calcs.allIsZero = false;
  }
  if (calcs.max !== null && calcs.min !== null) {
    calcs.range = calcs.max - calcs.min;
  }
  if (isNumber(calcs.firstNotNull) && isNumber(calcs.lastNotNull)) {
    calcs.diff = calcs.lastNotNull - calcs.firstNotNull;
  }
  if (isNumber(calcs.firstNotNull) && isNumber(calcs.diff)) {
    calcs.diffperc = calcs.diff / calcs.firstNotNull;
  }
  return calcs;
}
function calculateFirst(field, ignoreNulls, nullAsZero) {
  return { first: field.values.get(0) };
}
function calculateFirstNotNull(field, ignoreNulls, nullAsZero) {
  const data = field.values;
  for (let idx = 0; idx < data.length; idx++) {
    const v = data.get(idx);
    if (v != null && v !== void 0) {
      return { firstNotNull: v };
    }
  }
  return { firstNotNull: null };
}
function calculateLast(field, ignoreNulls, nullAsZero) {
  const data = field.values;
  return { last: data.get(data.length - 1) };
}
function calculateLastNotNull(field, ignoreNulls, nullAsZero) {
  const data = field.values;
  let idx = data.length - 1;
  while (idx >= 0) {
    const v = data.get(idx--);
    if (v != null && v !== void 0) {
      return { lastNotNull: v };
    }
  }
  return { lastNotNull: null };
}
function calculateStdDev(field, ignoreNulls, nullAsZero) {
  if (!(field.type === FieldType.number || field.type === FieldType.time)) {
    return { variance: 0, stdDev: 0 };
  }
  let squareSum = 0;
  let runningMean = 0;
  let runningNonNullCount = 0;
  const data = field.values;
  for (let i = 0; i < data.length; i++) {
    const currentValue = data.get(i);
    if (currentValue != null) {
      runningNonNullCount++;
      let _oldMean = runningMean;
      runningMean += (currentValue - _oldMean) / runningNonNullCount;
      squareSum += (currentValue - _oldMean) * (currentValue - runningMean);
    }
  }
  if (runningNonNullCount > 0) {
    const variance = squareSum / runningNonNullCount;
    return { variance, stdDev: Math.sqrt(variance) };
  }
  return { variance: 0, stdDev: 0 };
}
function calculateChangeCount(field, ignoreNulls, nullAsZero) {
  const data = field.values;
  let count = 0;
  let first = true;
  let last = null;
  for (let i = 0; i < data.length; i++) {
    let currentValue = data.get(i);
    if (currentValue === null) {
      if (ignoreNulls) {
        continue;
      }
      if (nullAsZero) {
        currentValue = 0;
      }
    }
    if (!first && last !== currentValue) {
      count++;
    }
    first = false;
    last = currentValue;
  }
  return { changeCount: count };
}
function calculateDistinctCount(field, ignoreNulls, nullAsZero) {
  const data = field.values;
  const distinct = /* @__PURE__ */ new Set();
  for (let i = 0; i < data.length; i++) {
    let currentValue = data.get(i);
    if (currentValue === null) {
      if (ignoreNulls) {
        continue;
      }
      if (nullAsZero) {
        currentValue = 0;
      }
    }
    distinct.add(currentValue);
  }
  return { distinctCount: distinct.size };
}

export { ReducerID, doStandardCalcs, fieldReducers, reduceField };
//# sourceMappingURL=fieldReducer.js.map
