import formatDuration from 'date-fns/formatDuration';
import React__default, { PureComponent } from 'react';
import { parseDuration } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { t } from '../../utils/i18n.js';
import '../Button/Button.js';
import { ButtonGroup } from '../Button/ButtonGroup.js';
import { ButtonSelect } from '../Dropdown/ButtonSelect.js';
import { ToolbarButton } from '../ToolbarButton/ToolbarButton.js';
import '../ToolbarButton/ToolbarButtonRow.js';

const defaultIntervals = ["5s", "10s", "30s", "1m", "5m", "15m", "30m", "1h", "2h", "1d"];
const _RefreshPicker = class extends PureComponent {
  constructor(props) {
    super(props);
    this.onChangeSelect = (item) => {
      const { onIntervalChanged } = this.props;
      if (onIntervalChanged && item.value != null) {
        onIntervalChanged(item.value);
      }
    };
  }
  getVariant() {
    if (this.props.isLive) {
      return "primary";
    }
    if (this.props.isLoading) {
      return "destructive";
    }
    if (this.props.primary) {
      return "primary";
    }
    return this.props.isOnCanvas ? "canvas" : "default";
  }
  render() {
    const { onRefresh, intervals, tooltip, value, text, isLoading, noIntervalPicker, width } = this.props;
    const currentValue = value || "";
    const variant = this.getVariant();
    const options = intervalsToOptions({ intervals });
    const option = options.find(({ value: value2 }) => value2 === currentValue);
    const translatedOffOption = translateOption(_RefreshPicker.offOption.value);
    let selectedValue = option || translatedOffOption;
    if (selectedValue.label === translatedOffOption.label) {
      selectedValue = { value: "" };
    }
    const durationAriaLabel = selectedValue.ariaLabel;
    const ariaLabelDurationSelectedMessage = t(
      "refresh-picker.aria-label.duration-selected",
      "Choose refresh time interval with current interval {{durationAriaLabel}} selected",
      { durationAriaLabel }
    );
    const ariaLabelChooseIntervalMessage = t(
      "refresh-picker.aria-label.choose-interval",
      "Auto refresh turned off. Choose refresh time interval"
    );
    const ariaLabel = selectedValue.value === "" ? ariaLabelChooseIntervalMessage : ariaLabelDurationSelectedMessage;
    return /* @__PURE__ */ React__default.createElement(ButtonGroup, { className: "refresh-picker" }, /* @__PURE__ */ React__default.createElement(
      ToolbarButton,
      {
        "aria-label": text,
        tooltip,
        onClick: onRefresh,
        variant,
        icon: isLoading ? "fa fa-spinner" : "sync",
        style: width ? { width } : void 0,
        "data-testid": selectors.components.RefreshPicker.runButtonV2
      },
      text
    ), !noIntervalPicker && /* @__PURE__ */ React__default.createElement(
      ButtonSelect,
      {
        value: selectedValue,
        options,
        onChange: this.onChangeSelect,
        variant,
        title: t("refresh-picker.select-button.auto-refresh", "Set auto refresh interval"),
        "data-testid": selectors.components.RefreshPicker.intervalButtonV2,
        "aria-label": ariaLabel
      }
    ));
  }
};
let RefreshPicker = _RefreshPicker;
RefreshPicker.offOption = {
  label: "Off",
  value: "",
  ariaLabel: "Turn off auto refresh"
};
RefreshPicker.liveOption = {
  label: "Live",
  value: "LIVE",
  ariaLabel: "Turn on live streaming"
};
RefreshPicker.isLive = (refreshInterval) => refreshInterval === _RefreshPicker.liveOption.value;
function translateOption(option) {
  if (option === RefreshPicker.liveOption.value) {
    return {
      label: t("refresh-picker.live-option.label", "Live"),
      value: "LIVE",
      ariaLabel: t("refresh-picker.live-option.aria-label", "Turn on live streaming")
    };
  }
  return {
    label: t("refresh-picker.off-option.label", "Off"),
    value: "",
    ariaLabel: t("refresh-picker.off-option.aria-label", "Turn off auto refresh")
  };
}
function intervalsToOptions({ intervals = defaultIntervals } = {}) {
  const options = intervals.map((interval) => {
    const duration = parseDuration(interval);
    const ariaLabel = formatDuration(duration);
    return {
      label: interval,
      value: interval,
      ariaLabel
    };
  });
  options.unshift(translateOption(RefreshPicker.offOption.value));
  return options;
}

export { RefreshPicker, defaultIntervals, intervalsToOptions, translateOption };
//# sourceMappingURL=RefreshPicker.js.map
