import { css, cx } from '@emotion/css';
import React__default, { useState } from 'react';
import { getDefaultTimeRange, dateMath, dateTime } from '@grafana/data';
import { selectors } from '@grafana/e2e-selectors';
import { useTheme2 } from '../../themes/ThemeContext.js';
import { stylesFactory } from '../../themes/stylesFactory.js';
import '@emotion/react';
import 'tinycolor2';
import { ClickOutsideWrapper } from '../ClickOutsideWrapper/ClickOutsideWrapper.js';
import { Icon } from '../Icon/Icon.js';
import { getInputStyles } from '../Input/Input.js';
import { TimePickerButtonLabel } from './TimeRangePicker.js';
import { TimePickerContent } from './TimeRangePicker/TimePickerContent.js';
import { quickOptions } from './options.js';

const isValidTimeRange = (range) => {
  return dateMath.isValid(range.from) && dateMath.isValid(range.to);
};
const noop = () => {
};
const TimeRangeInput = ({
  value,
  onChange,
  onChangeTimeZone = noop,
  clearable,
  hideTimeZone = true,
  timeZone = "browser",
  placeholder = "Select time range",
  isReversed = true,
  hideQuickRanges = false,
  disabled = false
}) => {
  const [isOpen, setIsOpen] = useState(false);
  const theme = useTheme2();
  const styles = getStyles(theme, disabled);
  const onOpen = (event) => {
    event.stopPropagation();
    event.preventDefault();
    if (disabled) {
      return;
    }
    setIsOpen(!isOpen);
  };
  const onClose = () => {
    setIsOpen(false);
  };
  const onRangeChange = (timeRange) => {
    onClose();
    onChange(timeRange);
  };
  const onRangeClear = (event) => {
    event.stopPropagation();
    const from = dateTime(null);
    const to = dateTime(null);
    onChange({ from, to, raw: { from, to } });
  };
  return /* @__PURE__ */ React__default.createElement("div", { className: styles.container }, /* @__PURE__ */ React__default.createElement(
    "button",
    {
      type: "button",
      className: styles.pickerInput,
      "aria-label": selectors.components.TimePicker.openButton,
      onClick: onOpen
    },
    isValidTimeRange(value) ? /* @__PURE__ */ React__default.createElement(TimePickerButtonLabel, { value, timeZone }) : /* @__PURE__ */ React__default.createElement("span", { className: styles.placeholder }, placeholder),
    !disabled && /* @__PURE__ */ React__default.createElement("span", { className: styles.caretIcon }, isValidTimeRange(value) && clearable && /* @__PURE__ */ React__default.createElement(Icon, { className: styles.clearIcon, name: "times", size: "lg", onClick: onRangeClear }), /* @__PURE__ */ React__default.createElement(Icon, { name: isOpen ? "angle-up" : "angle-down", size: "lg" }))
  ), isOpen && /* @__PURE__ */ React__default.createElement(ClickOutsideWrapper, { includeButtonPress: false, onClick: onClose }, /* @__PURE__ */ React__default.createElement(
    TimePickerContent,
    {
      timeZone,
      value: isValidTimeRange(value) ? value : getDefaultTimeRange(),
      onChange: onRangeChange,
      quickOptions,
      onChangeTimeZone,
      className: styles.content,
      hideTimeZone,
      isReversed,
      hideQuickRanges
    }
  )));
};
const getStyles = stylesFactory((theme, disabled = false) => {
  const inputStyles = getInputStyles({ theme, invalid: false });
  return {
    container: css`
      display: flex;
      position: relative;
    `,
    content: css`
      margin-left: 0;
      position: absolute;
      top: 116%;
      z-index: ${theme.zIndex.dropdown};
    `,
    pickerInput: cx(
      inputStyles.input,
      disabled && inputStyles.inputDisabled,
      inputStyles.wrapper,
      css`
        display: flex;
        align-items: center;
        justify-content: space-between;
        cursor: pointer;
        padding-right: 0;
        line-height: ${theme.spacing.gridSize * 4 - 2}px;
      `
    ),
    caretIcon: cx(
      inputStyles.suffix,
      css`
        position: relative;
        top: -1px;
        margin-left: ${theme.spacing(0.5)};
      `
    ),
    clearIcon: css`
      margin-right: ${theme.spacing(0.5)};
      &:hover {
        color: ${theme.colors.text.maxContrast};
      }
    `,
    placeholder: css`
      color: ${theme.colors.text.disabled};
      opacity: 1;
    `
  };
});

export { TimeRangeInput };
//# sourceMappingURL=TimeRangeInput.js.map
