import { defaults } from 'lodash';
import { map } from 'rxjs/operators';
import { getTimeField } from '../../dataframe/processDataFrame.js';
import { FieldMatcherID } from '../matchers/ids.js';
import { DataTransformerID } from './ids.js';
import { getFieldMatcher } from '../matchers.js';
import './concat.js';
import './convertFieldType.js';
import { ensureColumnsTransformer } from './ensureColumns.js';
import './filter.js';
import './filterByName.js';
import './filterByRefId.js';
import './filterByValue.js';
import './groupBy.js';
import './groupingToMatrix.js';
import './histogram.js';
import './joinByField.js';
import './labelsToFields.js';
import './limit.js';
import './merge.js';
import { noopTransformer } from './noop.js';
import './order.js';
import './organize.js';
import './reduce.js';
import './rename.js';
import './renameByRegex.js';
import './seriesToRows.js';
import './sortBy.js';
import { ReducerID, fieldReducers, doStandardCalcs } from '../fieldReducer.js';
import 'rxjs';
import '../standardTransformersRegistry.js';
import '../matchers/nameMatcher.js';
import 'fast_array_intersect';
import { FieldType } from '../../types/dataFrame.js';
import { ArrayVector } from '../../vector/ArrayVector.js';
import { NullValueMode } from '../../types/data.js';
import '@grafana/schema';
import '../../datetime/moment_wrapper.js';
import '../../types/datasource.js';
import '../../types/legacyEvents.js';
import { ConstantVector } from '../../vector/ConstantVector.js';
import { BinaryOperationVector } from '../../vector/BinaryOperationVector.js';
import '../../datetime/rangeutil.js';
import '../../datetime/timezones.js';
import '../../datetime/formats.js';
import 'moment-timezone';
import 'date-fns';
import 'date-fns/add';
import 'date-fns/intervalToDuration';
import { IndexVector } from '../../vector/IndexVector.js';
import { AsNumberVector } from '../../vector/AsNumberVector.js';
import '../../field/fieldColor.js';
import { getFieldDisplayName } from '../../field/fieldState.js';
import '../../field/standardFieldConfigEditorRegistry.js';
import 'react';
import 'react-use/lib/usePrevious';
import 'tinycolor2';
import 'papaparse';
import { BinaryOperationID, binaryOperators } from '../../utils/binaryOperators.js';
import 'marked';
import '../../text/sanitize.js';
import { RowVector } from '../../vector/RowVector.js';

var __defProp = Object.defineProperty;
var __defProps = Object.defineProperties;
var __getOwnPropDescs = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps = (a, b) => __defProps(a, __getOwnPropDescs(b));
const defaultReduceOptions = {
  reducer: ReducerID.sum
};
const defaultBinaryOptions = {
  left: "",
  operator: BinaryOperationID.Add,
  right: ""
};
const calculateFieldTransformer = {
  id: DataTransformerID.calculateField,
  name: "Add field from calculation",
  description: "Use the row values to calculate a new field",
  defaultOptions: {
    mode: "reduceRow" /* ReduceRow */,
    reduce: {
      reducer: ReducerID.sum
    }
  },
  operator: (options, ctx) => (outerSource) => {
    const operator = options && options.timeSeries !== false ? ensureColumnsTransformer.operator(null, ctx) : noopTransformer.operator({}, ctx);
    if (options.alias != null) {
      options.alias = ctx.interpolate(options.alias);
    }
    return outerSource.pipe(
      operator,
      map((data) => {
        var _a, _b, _c;
        const mode = (_a = options.mode) != null ? _a : "reduceRow" /* ReduceRow */;
        let creator = void 0;
        if (mode === "reduceRow" /* ReduceRow */) {
          creator = getReduceRowCreator(defaults(options.reduce, defaultReduceOptions), data);
        } else if (mode === "binary" /* BinaryOperation */) {
          const binaryOptions = __spreadProps(__spreadValues({}, options.binary), {
            left: ctx.interpolate((_b = options.binary) == null ? void 0 : _b.left),
            right: ctx.interpolate((_c = options.binary) == null ? void 0 : _c.right)
          });
          creator = getBinaryCreator(defaults(binaryOptions, defaultBinaryOptions), data);
        } else if (mode === "index" /* Index */) {
          return data.map((frame) => {
            var _a2;
            const f = {
              name: (_a2 = options.alias) != null ? _a2 : "Row",
              type: FieldType.number,
              values: new IndexVector(frame.length),
              config: {
                min: 0,
                max: frame.length - 1
              }
            };
            return __spreadProps(__spreadValues({}, frame), {
              fields: options.replaceFields ? [f] : [...frame.fields, f]
            });
          });
        }
        if (!creator) {
          return data;
        }
        return data.map((frame) => {
          const values = creator(frame);
          if (!values) {
            return frame;
          }
          const field = {
            name: getNameFromOptions(options),
            type: FieldType.number,
            config: {},
            values
          };
          let fields = [];
          if (options.replaceFields) {
            const { timeField } = getTimeField(frame);
            if (timeField && options.timeSeries !== false) {
              fields = [timeField, field];
            } else {
              fields = [field];
            }
          } else {
            fields = [...frame.fields, field];
          }
          return __spreadProps(__spreadValues({}, frame), {
            fields
          });
        });
      })
    );
  }
};
function getReduceRowCreator(options, allFrames) {
  var _a;
  let matcher = getFieldMatcher({
    id: FieldMatcherID.numeric
  });
  if (options.include && options.include.length) {
    matcher = getFieldMatcher({
      id: FieldMatcherID.byNames,
      options: {
        names: options.include
      }
    });
  }
  const info = fieldReducers.get(options.reducer);
  if (!info) {
    throw new Error(`Unknown reducer: ${options.reducer}`);
  }
  const reducer = (_a = info.reduce) != null ? _a : doStandardCalcs;
  const ignoreNulls = options.nullValueMode === NullValueMode.Ignore;
  const nullAsZero = options.nullValueMode === NullValueMode.AsZero;
  return (frame) => {
    const columns = [];
    for (const field of frame.fields) {
      if (matcher(field, frame, allFrames)) {
        columns.push(field.values);
      }
    }
    const iter = new RowVector(columns);
    const row = {
      name: "temp",
      values: iter,
      type: FieldType.number,
      config: {}
    };
    const vals = [];
    for (let i = 0; i < frame.length; i++) {
      iter.rowIndex = i;
      const val = reducer(row, ignoreNulls, nullAsZero)[options.reducer];
      vals.push(val);
    }
    return new ArrayVector(vals);
  };
}
function findFieldValuesWithNameOrConstant(frame, name, allFrames) {
  if (!name) {
    return void 0;
  }
  for (const f of frame.fields) {
    if (name === getFieldDisplayName(f, frame, allFrames)) {
      if (f.type === FieldType.boolean) {
        return new AsNumberVector(f.values);
      }
      return f.values;
    }
  }
  const v = parseFloat(name);
  if (!isNaN(v)) {
    return new ConstantVector(v, frame.length);
  }
  return void 0;
}
function getBinaryCreator(options, allFrames) {
  const operator = binaryOperators.getIfExists(options.operator);
  return (frame) => {
    const left = findFieldValuesWithNameOrConstant(frame, options.left, allFrames);
    const right = findFieldValuesWithNameOrConstant(frame, options.right, allFrames);
    if (!left || !right || !operator) {
      return void 0;
    }
    return new BinaryOperationVector(left, right, operator.operation);
  };
}
function getNameFromOptions(options) {
  var _a, _b, _c, _d, _e;
  if ((_a = options.alias) == null ? void 0 : _a.length) {
    return options.alias;
  }
  switch (options.mode) {
    case "binary" /* BinaryOperation */: {
      const { binary } = options;
      return `${(_b = binary == null ? void 0 : binary.left) != null ? _b : ""} ${(_c = binary == null ? void 0 : binary.operator) != null ? _c : ""} ${(_d = binary == null ? void 0 : binary.right) != null ? _d : ""}`;
    }
    case "reduceRow" /* ReduceRow */:
      {
        const r = fieldReducers.getIfExists((_e = options.reduce) == null ? void 0 : _e.reducer);
        if (r) {
          return r.name;
        }
      }
      break;
    case "index" /* Index */:
      return "Row";
  }
  return "math";
}

export { calculateFieldTransformer, getNameFromOptions };
//# sourceMappingURL=calculateField.js.map
