import { cx, css } from '@emotion/css';
import memoizeOne from 'memoize-one';
import React__default, { PureComponent } from 'react';
import Highlighter from 'react-highlight-words';
import tinycolor from 'tinycolor2';
import { findHighlightChunksInText } from '@grafana/data';
import { withTheme2 } from '../../themes/ThemeContext.js';
import '@emotion/react';
import { IconButton } from '../IconButton/IconButton.js';
import { Tooltip } from '../Tooltip/Tooltip.js';
import { LogMessageAnsi } from './LogMessageAnsi.js';
import { LogRowContext } from './LogRowContext.js';
import { getLogRowStyles } from './getLogRowStyles.js';

const MAX_CHARACTERS = 1e5;
const getStyles = (theme) => {
  const outlineColor = tinycolor(theme.components.dashboard.background).setAlpha(0.7).toRgbString();
  return {
    positionRelative: css`
      label: positionRelative;
      position: relative;
    `,
    rowWithContext: css`
      label: rowWithContext;
      z-index: 1;
      outline: 9999px solid ${outlineColor};
    `,
    horizontalScroll: css`
      label: verticalScroll;
      white-space: pre;
    `,
    contextNewline: css`
      display: block;
      margin-left: 0px;
    `,
    contextButton: css`
      display: flex;
      flex-wrap: nowrap;
      flex-direction: row;
      align-content: flex-end;
      justify-content: space-evenly;
      align-items: center;
      position: absolute;
      right: -8px;
      top: 0;
      bottom: auto;
      width: 80px;
      height: 36px;
      background: ${theme.colors.background.primary};
      box-shadow: ${theme.shadows.z3};
      padding: ${theme.spacing(0, 0, 0, 0.5)};
      z-index: 100;
    `
  };
};
function renderLogMessage(hasAnsi, entry, highlights, highlightClassName) {
  const needsHighlighter = highlights && highlights.length > 0 && highlights[0] && highlights[0].length > 0 && entry.length < MAX_CHARACTERS;
  const searchWords = highlights != null ? highlights : [];
  if (hasAnsi) {
    const highlight = needsHighlighter ? { searchWords, highlightClassName } : void 0;
    return /* @__PURE__ */ React__default.createElement(LogMessageAnsi, { value: entry, highlight });
  } else if (needsHighlighter) {
    return /* @__PURE__ */ React__default.createElement(
      Highlighter,
      {
        textToHighlight: entry,
        searchWords,
        findChunks: findHighlightChunksInText,
        highlightClassName
      }
    );
  } else {
    return entry;
  }
}
const restructureLog = memoizeOne((line, prettifyLogMessage) => {
  if (prettifyLogMessage) {
    try {
      return JSON.stringify(JSON.parse(line), void 0, 2);
    } catch (error) {
      return line;
    }
  }
  return line;
});
class UnThemedLogRowMessage extends PureComponent {
  constructor() {
    super(...arguments);
    this.onContextToggle = (e) => {
      e.stopPropagation();
      this.props.onToggleContext();
    };
  }
  render() {
    const {
      row,
      theme,
      errors,
      hasMoreContextRows,
      updateLimit,
      context,
      contextIsOpen,
      showContextToggle,
      wrapLogMessage,
      prettifyLogMessage,
      onToggleContext
    } = this.props;
    const style = getLogRowStyles(theme, row.logLevel);
    const { hasAnsi, raw } = row;
    const restructuredEntry = restructureLog(raw, prettifyLogMessage);
    const styles = getStyles(theme);
    return (
      // When context is open, the position has to be NOT relative.
      // Setting the postion as inline-style to overwrite the more sepecific style definition from `style.logsRowMessage`.
      /* @__PURE__ */ React__default.createElement("td", { style: contextIsOpen ? { position: "unset" } : void 0, className: style.logsRowMessage }, /* @__PURE__ */ React__default.createElement(
        "div",
        {
          className: cx({ [styles.positionRelative]: wrapLogMessage }, { [styles.horizontalScroll]: !wrapLogMessage })
        },
        contextIsOpen && context && /* @__PURE__ */ React__default.createElement(
          LogRowContext,
          {
            row,
            context,
            errors,
            wrapLogMessage,
            hasMoreContextRows,
            onOutsideClick: onToggleContext,
            onLoadMoreContext: () => {
              if (updateLimit) {
                updateLimit();
              }
            }
          }
        ),
        /* @__PURE__ */ React__default.createElement("span", { className: cx(styles.positionRelative, { [styles.rowWithContext]: contextIsOpen }) }, renderLogMessage(hasAnsi, restructuredEntry, row.searchWords, style.logsRowMatchHighLight)),
        !contextIsOpen && (showContextToggle == null ? void 0 : showContextToggle(row)) && /* @__PURE__ */ React__default.createElement(
          "span",
          {
            className: cx("log-row-context", style.context, styles.contextButton),
            onClick: (e) => e.stopPropagation()
          },
          /* @__PURE__ */ React__default.createElement(Tooltip, { placement: "top", content: "Show context" }, /* @__PURE__ */ React__default.createElement(IconButton, { size: "md", name: "gf-show-context", onClick: this.onContextToggle })),
          /* @__PURE__ */ React__default.createElement(Tooltip, { placement: "top", content: "Copy" }, /* @__PURE__ */ React__default.createElement(
            IconButton,
            {
              size: "md",
              name: "copy",
              onClick: () => navigator.clipboard.writeText(JSON.stringify(restructuredEntry))
            }
          ))
        )
      ))
    );
  }
}
const LogRowMessage = withTheme2(UnThemedLogRowMessage);
LogRowMessage.displayName = "LogRowMessage";

export { LogRowMessage, MAX_CHARACTERS };
//# sourceMappingURL=LogRowMessage.js.map
