"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFileLoaders = exports.getStyleLoaders = exports.getStylesheetEntries = void 0;
const tslib_1 = require("tslib");
const fs_1 = tslib_1.__importDefault(require("fs"));
const path_1 = tslib_1.__importDefault(require("path"));
const getPluginId_1 = require("../utils/getPluginId");
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const supportedExtensions = ['css', 'scss', 'less', 'sass'];
const getStylesheetPaths = (root = process.cwd()) => {
    return [`${root}/src/styles/light`, `${root}/src/styles/dark`];
};
const getStylesheetEntries = (root = process.cwd()) => {
    const stylesheetsPaths = getStylesheetPaths(root);
    const entries = {};
    supportedExtensions.forEach((e) => {
        stylesheetsPaths.forEach((p) => {
            const entryName = p.split('/').slice(-1)[0];
            if (fs_1.default.existsSync(`${p}.${e}`)) {
                if (entries[entryName]) {
                    console.log(`\nSeems like you have multiple files for ${entryName} theme:`);
                    console.log(entries[entryName]);
                    console.log(`${p}.${e}`);
                    throw new Error('Duplicated stylesheet');
                }
                else {
                    entries[entryName] = `${p}.${e}`;
                }
            }
        });
    });
    return entries;
};
exports.getStylesheetEntries = getStylesheetEntries;
const getStyleLoaders = () => {
    const extractionLoader = {
        loader: MiniCssExtractPlugin.loader,
        options: {
            publicPath: '../',
        },
    };
    const cssLoaders = [
        {
            loader: require.resolve('css-loader'),
            options: {
                importLoaders: 1,
                sourceMap: true,
            },
        },
        {
            loader: require.resolve('postcss-loader'),
            options: {
                postcssOptions: {
                    plugins: () => [
                        require('postcss-flexbugs-fixes'),
                        require('postcss-preset-env')({
                            autoprefixer: { flexbox: 'no-2009', grid: true },
                        }),
                    ],
                },
            },
        },
    ];
    const styleDir = path_1.default.resolve(process.cwd(), 'src', 'styles') + path_1.default.sep;
    const rules = [
        {
            test: /(dark|light)\.css$/,
            use: [extractionLoader, ...cssLoaders],
        },
        {
            test: /(dark|light)\.scss$/,
            use: [extractionLoader, ...cssLoaders, require.resolve('sass-loader')],
        },
        {
            test: /\.css$/,
            use: ['style-loader', ...cssLoaders, require.resolve('sass-loader')],
            exclude: [`${styleDir}light.css`, `${styleDir}dark.css`],
        },
        {
            test: /\.s[ac]ss$/,
            use: ['style-loader', ...cssLoaders, require.resolve('sass-loader')],
            exclude: [`${styleDir}light.scss`, `${styleDir}dark.scss`],
        },
        {
            test: /\.less$/,
            use: [
                {
                    loader: require.resolve('style-loader'),
                },
                ...cssLoaders,
                {
                    loader: require.resolve('less-loader'),
                    options: {
                        lessOptions: {
                            javascriptEnabled: true,
                        },
                    },
                },
            ],
            exclude: [`${styleDir}light.less`, `${styleDir}dark.less`],
        },
    ];
    return rules;
};
exports.getStyleLoaders = getStyleLoaders;
const getFileLoaders = () => {
    return [
        {
            test: /\.(png|jpe?g|gif|svg)$/,
            type: 'asset/resource',
            generator: {
                publicPath: `public/plugins/${(0, getPluginId_1.getPluginId)()}/img/`,
                outputPath: 'img/',
            },
        },
        {
            test: /\.(woff|woff2|eot|ttf|otf)(\?v=\d+\.\d+\.\d+)?$/,
            type: 'asset/resource',
            generator: {
                publicPath: `public/plugins/${(0, getPluginId_1.getPluginId)()}/fonts/`,
                outputPath: 'fonts/',
            },
        },
    ];
};
exports.getFileLoaders = getFileLoaders;
//# sourceMappingURL=loaders.js.map