import { escapeRegExp, countBy, chain } from 'lodash';
import { FieldType } from '../types/dataFrame.js';
import { LogLevel } from '../types/logs.js';
import { LogsSortOrder } from '@grafana/schema';
import '../datetime/moment_wrapper.js';
import '../types/datasource.js';
import '../types/legacyEvents.js';
import { ArrayVector } from '../vector/ArrayVector.js';

var __defProp = Object.defineProperty;
var __defProps = Object.defineProperties;
var __getOwnPropDescs = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps = (a, b) => __defProps(a, __getOwnPropDescs(b));
const LOGFMT_REGEXP = /(?:^|\s)([\w\(\)\[\]\{\}]+)=(""|(?:".*?[^\\]"|[^"\s]\S*))/;
function getLogLevel(line) {
  if (!line) {
    return LogLevel.unknown;
  }
  let level = LogLevel.unknown;
  let currentIndex = void 0;
  for (const key of Object.keys(LogLevel)) {
    const regexp = new RegExp(`\\b${key}\\b`, "i");
    const result = regexp.exec(line);
    if (result) {
      if (currentIndex === void 0 || result.index < currentIndex) {
        level = LogLevel[key];
        currentIndex = result.index;
      }
    }
  }
  return level;
}
function getLogLevelFromKey(key) {
  const level = LogLevel[key.toString().toLowerCase()];
  if (level) {
    return level;
  }
  return LogLevel.unknown;
}
function addLogLevelToSeries(series, lineIndex) {
  const levels = new ArrayVector();
  const lines = series.fields[lineIndex];
  for (let i = 0; i < lines.values.length; i++) {
    const line = lines.values.get(lineIndex);
    levels.buffer.push(getLogLevel(line));
  }
  return __spreadProps(__spreadValues({}, series), {
    // Keeps Tags, RefID etc
    fields: [
      ...series.fields,
      {
        name: "LogLevel",
        type: FieldType.string,
        values: levels,
        config: {}
      }
    ]
  });
}
const LogsParsers = {
  JSON: {
    buildMatcher: (label) => new RegExp(`(?:{|,)\\s*"${label}"\\s*:\\s*"?([\\d\\.]+|[^"]*)"?`),
    getFields: (line) => {
      try {
        const parsed = JSON.parse(line);
        return Object.keys(parsed).map((key) => {
          return `"${key}":${JSON.stringify(parsed[key])}`;
        });
      } catch (e) {
      }
      return [];
    },
    getLabelFromField: (field) => (field.match(/^"([^"]+)"\s*:/) || [])[1],
    getValueFromField: (field) => (field.match(/:\s*(.*)$/) || [])[1],
    test: (line) => {
      let parsed;
      try {
        parsed = JSON.parse(line);
      } catch (error) {
      }
      return typeof parsed === "object";
    }
  },
  logfmt: {
    buildMatcher: (label) => new RegExp(`(?:^|\\s)${escapeRegExp(label)}=("[^"]*"|\\S+)`),
    getFields: (line) => {
      const fields = [];
      line.replace(new RegExp(LOGFMT_REGEXP, "g"), (substring) => {
        fields.push(substring.trim());
        return "";
      });
      return fields;
    },
    getLabelFromField: (field) => (field.match(LOGFMT_REGEXP) || [])[1],
    getValueFromField: (field) => (field.match(LOGFMT_REGEXP) || [])[2],
    test: (line) => LOGFMT_REGEXP.test(line)
  }
};
function calculateFieldStats(rows, extractor) {
  const rowsWithField = rows.filter((row) => extractor.test(row.entry));
  const rowCount = rowsWithField.length;
  const countsByValue = countBy(rowsWithField, (r) => {
    const row = r;
    const match = row.entry.match(extractor);
    return match ? match[1] : null;
  });
  return getSortedCounts(countsByValue, rowCount);
}
function calculateLogsLabelStats(rows, label) {
  const rowsWithLabel = rows.filter((row) => row.labels[label] !== void 0);
  const rowCount = rowsWithLabel.length;
  const countsByValue = countBy(rowsWithLabel, (row) => row.labels[label]);
  return getSortedCounts(countsByValue, rowCount);
}
function calculateStats(values) {
  const nonEmptyValues = values.filter((value) => value !== void 0 && value !== null);
  const countsByValue = countBy(nonEmptyValues);
  return getSortedCounts(countsByValue, nonEmptyValues.length);
}
const getSortedCounts = (countsByValue, rowCount) => {
  return chain(countsByValue).map((count, value) => ({ count, value, proportion: count / rowCount })).sortBy("count").reverse().value();
};
function getParser(line) {
  let parser;
  try {
    if (LogsParsers.JSON.test(line)) {
      parser = LogsParsers.JSON;
    }
  } catch (error) {
  }
  if (!parser && LogsParsers.logfmt.test(line)) {
    parser = LogsParsers.logfmt;
  }
  return parser;
}
const sortInAscendingOrder = (a, b) => {
  if (a.timeEpochMs < b.timeEpochMs) {
    return -1;
  }
  if (a.timeEpochMs > b.timeEpochMs) {
    return 1;
  }
  if (a.timeEpochNs < b.timeEpochNs) {
    return -1;
  }
  if (a.timeEpochNs > b.timeEpochNs) {
    return 1;
  }
  return 0;
};
const sortInDescendingOrder = (a, b) => {
  if (a.timeEpochMs > b.timeEpochMs) {
    return -1;
  }
  if (a.timeEpochMs < b.timeEpochMs) {
    return 1;
  }
  if (a.timeEpochNs > b.timeEpochNs) {
    return -1;
  }
  if (a.timeEpochNs < b.timeEpochNs) {
    return 1;
  }
  return 0;
};
const sortLogsResult = (logsResult, sortOrder) => {
  const rows = logsResult ? sortLogRows(logsResult.rows, sortOrder) : [];
  return logsResult ? __spreadProps(__spreadValues({}, logsResult), { rows }) : { hasUniqueLabels: false, rows };
};
const sortLogRows = (logRows, sortOrder) => sortOrder === LogsSortOrder.Ascending ? logRows.sort(sortInAscendingOrder) : logRows.sort(sortInDescendingOrder);
const checkLogsError = (logRow) => {
  if (logRow.labels.__error__) {
    return {
      hasError: true,
      errorMessage: logRow.labels.__error__
    };
  }
  return {
    hasError: false
  };
};
const escapeUnescapedString = (string) => string.replace(/\\r\\n|\\n|\\t|\\r/g, (match) => match.slice(1) === "t" ? "	" : "\n");

export { LogsParsers, addLogLevelToSeries, calculateFieldStats, calculateLogsLabelStats, calculateStats, checkLogsError, escapeUnescapedString, getLogLevel, getLogLevelFromKey, getParser, sortInAscendingOrder, sortInDescendingOrder, sortLogRows, sortLogsResult };
//# sourceMappingURL=logs.js.map
