import { map } from 'rxjs/operators';
import { MutableDataFrame } from '../../dataframe/MutableDataFrame.js';
import 'lodash';
import '../../datetime/moment_wrapper.js';
import '../../datetime/rangeutil.js';
import '../../datetime/timezones.js';
import '../../datetime/formats.js';
import 'moment-timezone';
import '@grafana/schema';
import 'date-fns';
import 'date-fns/add';
import 'date-fns/intervalToDuration';
import { FieldType } from '../../types/dataFrame.js';
import { getFieldDisplayName } from '../../field/fieldState.js';
import { SpecialValue } from '../../types/transformations.js';
import '../../types/datasource.js';
import '../../types/legacyEvents.js';
import { DataTransformerID } from './ids.js';

const DEFAULT_COLUMN_FIELD = "Time";
const DEFAULT_ROW_FIELD = "Time";
const DEFAULT_VALUE_FIELD = "Value";
const DEFAULT_EMPTY_VALUE = SpecialValue.Empty;
const groupingToMatrixTransformer = {
  id: DataTransformerID.groupingToMatrix,
  name: "Grouping to Matrix",
  description: "Groups series by field and return a matrix visualisation",
  defaultOptions: {
    columnField: DEFAULT_COLUMN_FIELD,
    rowField: DEFAULT_ROW_FIELD,
    valueField: DEFAULT_VALUE_FIELD
  },
  operator: (options) => (source) => source.pipe(
    map((data) => {
      var _a;
      const columnFieldMatch = options.columnField || DEFAULT_COLUMN_FIELD;
      const rowFieldMatch = options.rowField || DEFAULT_ROW_FIELD;
      const valueFieldMatch = options.valueField || DEFAULT_VALUE_FIELD;
      const emptyValue = options.emptyValue || DEFAULT_EMPTY_VALUE;
      if (data.length !== 1) {
        return data;
      }
      const frame = data[0];
      const keyColumnField = findKeyField(frame, columnFieldMatch);
      const keyRowField = findKeyField(frame, rowFieldMatch);
      const valueField = findKeyField(frame, valueFieldMatch);
      const rowColumnField = `${rowFieldMatch}\\${columnFieldMatch}`;
      if (!keyColumnField || !keyRowField || !valueField) {
        return data;
      }
      const columnValues = uniqueValues(keyColumnField.values);
      const rowValues = uniqueValues(keyRowField.values);
      const matrixValues = {};
      for (let index = 0; index < valueField.values.length; index++) {
        const columnName = keyColumnField.values.get(index);
        const rowName = keyRowField.values.get(index);
        const value = valueField.values.get(index);
        if (!matrixValues[columnName]) {
          matrixValues[columnName] = {};
        }
        matrixValues[columnName][rowName] = value;
      }
      const resultFrame = new MutableDataFrame();
      resultFrame.addField({
        name: rowColumnField,
        values: rowValues,
        type: FieldType.string
      });
      for (const columnName of columnValues) {
        let values = [];
        for (const rowName of rowValues) {
          const value = (_a = matrixValues[columnName][rowName]) != null ? _a : getSpecialValue(emptyValue);
          values.push(value);
        }
        resultFrame.addField({
          name: columnName.toString(),
          values,
          config: valueField.config,
          type: valueField.type
        });
      }
      return [resultFrame];
    })
  )
};
function uniqueValues(values) {
  const unique = /* @__PURE__ */ new Set();
  for (let index = 0; index < values.length; index++) {
    unique.add(values.get(index));
  }
  return Array.from(unique);
}
function findKeyField(frame, matchTitle) {
  for (let fieldIndex = 0; fieldIndex < frame.fields.length; fieldIndex++) {
    const field = frame.fields[fieldIndex];
    if (matchTitle === getFieldDisplayName(field)) {
      return field;
    }
  }
  return null;
}
function getSpecialValue(specialValue) {
  switch (specialValue) {
    case SpecialValue.False:
      return false;
    case SpecialValue.True:
      return true;
    case SpecialValue.Null:
      return null;
    case SpecialValue.Empty:
    default:
      return "";
  }
}

export { groupingToMatrixTransformer };
//# sourceMappingURL=groupingToMatrix.js.map
